<?php
/**
 * This class manage the admin area
 *
 * @package YITH\DynamicPricingAndDiscounts\Classes
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Implements admin features of YITH WooCommerce Dynamic Pricing & Discounts
 *
 * @class   YITH_WC_Dynamic_Pricing_Admin
 * @package YITH WooCommerce Dynamic Pricing & Discounts
 * @since   1.0.0
 * @author  YITH <plugins@yithemes.com>
 */
if ( ! class_exists( 'YITH_WC_Dynamic_Pricing_Admin' ) ) {

	/**
	 * Class YITH_WC_Dynamic_Pricing_Admin
	 */
	class YITH_WC_Dynamic_Pricing_Admin {
		use YWDPD_Singleton_Trait;


		/**
		 * This is the instance of the YIT WooCommerce Panel class.
		 *
		 * @var YIT_Plugin_Panel_WooCommerce
		 */
		protected $panel;

		/**
		 * Panel page
		 *
		 * @var string
		 */
		protected $panel_page = 'yith_woocommerce_dynamic_pricing_and_discounts';


		/**
		 * Post type name.
		 *
		 * @var string Doc Url
		 */
		public $post_type_name = 'ywdpd_discount';

		/**
		 * Constructor
		 *
		 * Initialize plugin and registers actions and filters to be used
		 *
		 * @since  1.0.0
		 */
		public function __construct() {

			$plugin_basename = plugin_basename( YITH_YWDPD_DIR . '/' . basename( YITH_YWDPD_FILE ) );

			// register plugin to licence/update system.
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_activation' ), 99 );
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_updates' ),99 );
			// Add action links.
			add_filter(
				'plugin_action_links_' . $plugin_basename,
				array(
					$this,
					'action_links',
				)
			);
			add_filter( 'yith_show_plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 5 );
			add_action( 'admin_menu', array( $this, 'register_panel' ), 5 );
			add_action( 'admin_init', array( $this, 'add_metabox' ) );

			// Custom action for dynamic rule.
			add_action( 'admin_action_duplicate_discount', array( $this, 'duplicate_discount' ), 30 );
			add_action( 'admin_action_create_dynamic_rule', array( $this, 'create_dynamic_rule' ), 30 );
			add_action( 'admin_action_create_simple_dynamic_rule', array( $this, 'create_simple_dynamic_rule' ), 30 );
			add_filter( 'yith_plugin_fw_yith-plugins_page_yith_woocommerce_dynamic_pricing_and_discounts_bulk_delete_confirmation_enabled', '__return_true' );

			// Metabox customizations.
			add_filter( 'get_user_option_screen_layout_' . $this->post_type_name, '__return_true' );
			add_action( 'edit_form_top', array( $this, 'add_rule_info_badge' ), 11 );
			add_action( 'ywdpd_show_custom_field', array( $this, 'show_custom_field' ) );
			add_filter( 'yith_plugin_fw_inline_fields_allowed_types', array( $this, 'add_supported_types' ), 10, 3 );
			add_action( 'dbx_post_sidebar', array( $this, 'print_save_button_in_edit_page' ) );

			// Dynamic rule actions.
			add_action( 'save_post', array( $this, 'process_save_rule' ), 25, 2 );
			add_action( 'delete_post', array( $this, 'process_delete_rule' ), 20, 2 );
			add_action( 'wp_insert_post', array( $this, 'delete_transient' ), 20, 2 );
			add_filter( 'post_updated_messages', array( $this, 'post_updated_messages' ) );
			add_filter( 'bulk_post_updated_messages', array( $this, 'bulk_post_updated_messages' ), 10, 2 );

			// Exclusion list actions.
			add_action( 'admin_init', array( $this, 'delete_item_from_exclusion_list' ), 30 );
			add_action( 'admin_init', array( $this, 'add_items_in_exclusion_list' ), 20 );
			add_action( 'yith_ywdpd_exclusions_table', array( $this, 'show_the_exclusion_table' ) );

			// Other actions.
			add_action( 'admin_footer', array( $this, 'show_admin_popup' ), 25 );
			add_action( 'woocommerce_after_order_itemmeta', array( $this, 'show_custom_meta_item_in_order' ), 20, 3 );
			add_filter( 'woocommerce_order_item_get_code', array( $this, 'show_right_coupon_info' ), 20, 2 );

			// Admin notices.
			add_action( 'admin_notices', array( $this, 'print_notices' ) );

			add_action( 'save_post_ywdpd_discount', array( $this, 'invalidate_shipping_cache' ) );
		}

		/**
		 * Return the private property panel page.
		 *
		 * @return string
		 */
		public function get_panel_page() {
			return $this->panel_page;
		}


		/**
		 * Clone the rule
		 */
		public function duplicate_discount() {

			if ( empty( $_REQUEST['post'] ) ) {
				wp_die( esc_html__( 'No rule to duplicate has been supplied!', 'ywdpd' ) );
			}

			$rule_id = isset( $_REQUEST['post'] ) ? absint( $_REQUEST['post'] ) : '';

			check_admin_referer( 'ywdpd-duplicate-rule_' . $rule_id );

			global $wpdb;

			$query = $wpdb->prepare( "SELECT * from $wpdb->posts WHERE id=%d", $rule_id ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching

			$results        = $wpdb->get_results( $query, ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.PreparedSQL.NotPrepared
			$admin_post_url = admin_url( 'post.php' );
			$url_args       = array(
				'action' => 'edit',
			);
			if ( $results ) {
				foreach ( $results as $result ) {
					if ( 'ywdpd_discount' === $result['post_type'] ) {
						unset( $result['ID'] );
						$result['post_title'] .= ' ' . __( '(Copy)', 'ywdpd' );

						$new_post  = wp_insert_post( $result );
						$post_meta = get_post_custom( $rule_id );
						// set unique key and correct post id.
						$post_meta['_key'][0]       = uniqid();
						$post_meta['id'][0]         = $new_post;
						$post_meta['_priority'][0]  = ywdpd_get_last_priority() + 1;
						$post_meta['_rule_name'][0] = $result['post_title'];

						if ( is_array( $post_meta ) ) {
							foreach ( $post_meta as $k => $v ) {
								update_post_meta( $new_post, $k, maybe_unserialize( $v[0] ) );
							}
						}
						$url_args['post'] = $new_post;
						$admin_post_url   = esc_url_raw( add_query_arg( $url_args, $admin_post_url ) );
						wp_safe_redirect( $admin_post_url );
					}
				}
			}

			exit;
		}

		/**
		 * Add metabox into ywdpd_discount editor page
		 *
		 * @since  1.4.0
		 */
		public function add_metabox() {

			if ( ywdpd_check_valid_admin_page( $this->post_type_name ) ) {
				$post_rule_id = false;
				if ( isset( $_REQUEST['post'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$post_rule_id = sanitize_text_field( wp_unslash( $_REQUEST['post'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				} elseif ( isset( $_REQUEST['post_ID'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$post_rule_id = sanitize_text_field( wp_unslash( $_REQUEST['post_ID'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				}
				if ( $post_rule_id ) {
					$args = require_once YITH_YWDPD_INC . 'admin/metaboxes/dynamic-discount-metaboxes.php';

					if ( ! function_exists( 'YIT_Metabox' ) ) {
						require_once 'plugin-fw/yit-plugin.php';
					}

					$metabox = YIT_Metabox( 'ywdpd_dynamic_discount' );

					$metabox->init( $args );
					remove_meta_box( 'submitdiv', 'ywdpd_discount', 'side' );
				}
			}
		}

		/**
		 * Returns the panel page URI
		 *
		 * @param string $tab The current tab.
		 *
		 * @return string
		 *
		 * @since  1.4.0
		 */
		public function get_panel_page_uri( $tab = '' ) {
			$panel_uri = add_query_arg( 'page', $this->panel_page, admin_url( 'admin.php' ) );
			if ( $tab ) {
				$panel_uri = add_query_arg( 'tab', $tab, $panel_uri );
			}

			return $panel_uri;
		}

		/**
		 * Show the custom fields
		 *
		 * @param array $field The field.
		 *
		 * @return void
		 * @since 4.0.0
		 */
		public function show_custom_field( $field ) {
			if ( isset( $field['ywdpd_type'] ) ) {
				$type = $field['ywdpd_type'];

				$file_path = YITH_YWDPD_INC . '/admin/custom-fields/' . $type . '.php';
				if ( file_exists( $file_path ) ) {
					include $file_path;
				}
			}
		}

		/**
		 * Add in inline field custom supported types
		 *
		 * @param array  $types The default types.
		 * @param string $name  The field name.
		 * @param array  $field The field.
		 *
		 * @return array
		 * @since 4.0.0
		 */
		public function add_supported_types( $types, $name, $field ) {

			if ( strpos( $name, '_quantity_category_discount' ) !== false ) {
				$types[] = 'ajax-terms';
			}

			return $types;
		}

		/**
		 * Modify the capability
		 *
		 * @param string $capability Capability.
		 *
		 * @return string
		 */
		public function change_capability( $capability ) {
			return 'manage_woocommerce';
		}

		/**
		 * Add a panel under YITH Plugins tab
		 *
		 * @return   void
		 * @since    1.0
		 * @use      /Yit_Plugin_Panel class
		 * @see      plugin-fw/lib/yit-plugin-panel.php
		 */
		public function register_panel() {

			if ( ! empty( $this->panel ) ) {
				return;
			}

			$admin_tabs = array(
				'dynamic-rules' => array(
					'title'       => __( 'Dynamic Rules', 'ywdpd' ),
					'icon'        => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" stroke-width="1.5" stroke="currentColor">
<path d="M9 14.25L15 8.25M19.5 4.75699V21.75L15.75 20.25L12 21.75L8.25 20.25L4.5 21.75V4.75699C4.5 3.649 5.30608 2.70014 6.40668 2.57241C8.24156 2.35947 10.108 2.25 12 2.25C13.892 2.25 15.7584 2.35947 17.5933 2.57241C18.6939 2.70014 19.5 3.649 19.5 4.75699ZM9.75 9H9.7575V9.0075H9.75V9ZM10.125 9C10.125 9.20711 9.95711 9.375 9.75 9.375C9.54289 9.375 9.375 9.20711 9.375 9C9.375 8.79289 9.54289 8.625 9.75 8.625C9.95711 8.625 10.125 8.79289 10.125 9ZM14.25 13.5H14.2575V13.5075H14.25V13.5ZM14.625 13.5C14.625 13.7071 14.4571 13.875 14.25 13.875C14.0429 13.875 13.875 13.7071 13.875 13.5C13.875 13.2929 14.0429 13.125 14.25 13.125C14.4571 13.125 14.625 13.2929 14.625 13.5Z" stroke="#0F172A" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>
',
					'description' => __( 'A list of all deals, price rules and discounts created for your shop ', 'ywdpd' ),
				),
				'general'       => array(
					'title' => __( 'Settings', 'ywdpd' ),
					'icon'  => 'settings',
				),
				'exclusions'    => array(
					'title' => __( 'Exclusion List', 'ywdpd' ),
					'icon'  => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" stroke-width="1.5" stroke="currentColor">
<path d="M9 12H12.75M9 15H12.75M9 18H12.75M15.75 18.75H18C19.2426 18.75 20.25 17.7426 20.25 16.5V6.10822C20.25 4.97324 19.405 4.01015 18.2739 3.91627C17.9006 3.88529 17.5261 3.85858 17.1505 3.83619M11.3495 3.83619C11.2848 4.04602 11.25 4.26894 11.25 4.5C11.25 4.91421 11.5858 5.25 12 5.25H16.5C16.9142 5.25 17.25 4.91421 17.25 4.5C17.25 4.26894 17.2152 4.04602 17.1505 3.83619M11.3495 3.83619C11.6328 2.91757 12.4884 2.25 13.5 2.25H15C16.0116 2.25 16.8672 2.91757 17.1505 3.83619M11.3495 3.83619C10.9739 3.85858 10.5994 3.88529 10.2261 3.91627C9.09499 4.01015 8.25 4.97324 8.25 6.10822V8.25M8.25 8.25H4.875C4.25368 8.25 3.75 8.75368 3.75 9.375V20.625C3.75 21.2463 4.25368 21.75 4.875 21.75H14.625C15.2463 21.75 15.75 21.2463 15.75 20.625V9.375C15.75 8.75368 15.2463 8.25 14.625 8.25H8.25ZM6.75 12H6.7575V12.0075H6.75V12ZM6.75 15H6.7575V15.0075H6.75V15ZM6.75 18H6.7575V18.0075H6.75V18Z" stroke="#0F172A" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>
',
				),
			);

			$args = array(
				'create_menu_page' => true,
				'parent_slug'      => '',
				'plugin_slug'      => YITH_YWDPD_SLUG,
				'page_title'       => _x( 'YITH WooCommerce Dynamic Pricing & Discounts Premium', 'Plugin name, do not translate', 'ywdpd' ),
				'menu_title'       => 'Dynamic Pricing & Discounts',
				'capability'       => 'manage_options',
				'parent'           => 'ywdpd',
				'parent_page'      => 'yith_plugin_panel',
				'page'             => $this->panel_page,
				'admin-tabs'       => $admin_tabs,
				'options-path'     => YITH_YWDPD_DIR . '/plugin-options',
				'class'            => yith_set_wrapper_class(),
				'ui_version'       => 2,
				'help_tab'         => array(
					'main_video' => array(
						'desc' => _x( 'Check this video to learn how to <b>create dynamic rules</b>', '[HELP TAB] Video title', 'ywdpd' ),
						'url'  => array(
							'en' => '//youtube.com/embed/ayLe79SfXto',
							'it' => '//youtube.com/embed/Glrkj1UaD0g',
							'es' => '//youtube.com/embed/vjoAcaMkrL4',
						),
					),
					'playlists'  => array(
						'en' => '//youtube.com/watch?v=ayLe79SfXto&list=PLDriKG-6905n3DUEI6_S217zVl6yNoNzY&ab_channel=YITH',
						'it' => '//youtube.com/watch?v=Glrkj1UaD0g&list=PL9c19edGMs098ev6sVFZRtE7hLyxSaqaL&ab_channel=YITHITALIA',
						'es' => '//youtube.com/watch?v=vjoAcaMkrL4&list=PL9Ka3j92PYJPp5kYy3UV03PRv0VL4TZ1S&ab_channel=YITHESPA%C3%91A',
					),
					'hc_url'     => '//support.yithemes.com/hc/en-us/categories/4410223944209-YITH-WOOCOMMERCE-DYNAMIC-PRICING-AND-DISCOUNTS',
					'doc_url'    => '//docs.yithemes.com/yith-woocommerce-dynamic-pricing-and-discounts/',
				),
				'is_premium'       => true,
				'your_store_tools' => array(
					'items' => array(
						'membership'             => array(
							'name'           => 'Membership',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/membership.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-membership/',
							'description'    => _x(
								'Activate some sections of your e-commerce with restricted access so as to create memberships in your store.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Membership',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WCMBS_PREMIUM' ),
							'is_recommended' => true,
						),
						'multi-vendor'           => array(
							'name'           => 'Multi Vendor / Marketplace',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/multi-vendor.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-multi-vendor/',
							'description'    => _x(
								'Turn your e-commerce store into a marketplace (a multi-vendor platform) and earn commissions on orders generated by your vendors.',
								'[YOUR STORE TOOLS TAB] Description for plugin MultiVendor',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WPV_PREMIUM' ),
							'is_recommended' => true,
						),
						'brand-addon'            => array(
							'name'           => 'Brands Add-On',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/brands-addon.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-brands-add-on/',
							'description'    => _x(
								'Create unlimited brands to assign to your products to generate reliability and improve user experience by helping users easily find products of specific brands.',
								'[YOUR STORE TOOLS TAB] Description for plugin Brands Add-On',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WCBR_PREMIUM_INIT' ),
							'is_recommended' => true,
						),
						'wishlist'               => array(
							'name'           => 'Wishlist',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/wishlist.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-wishlist/',
							'description'    => _x( 'Allow your customers to create lists of products they want and share them with family and friends.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Wishlist',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WCWL_PREMIUM' ),
							'is_recommended' => true,
						),
						'ajax-product-filter'    => array(
							'name'           => 'Ajax Product Filter',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/ajax-product-filter.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-ajax-product-filter/',
							'description'    => _x(
								'Help your customers to easily find the products they are looking for and improve the user experience of your shop.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Ajax Product Filter',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WCAN_PREMIUM' ),
							'is_recommended' => true,
						),
						'booking'                => array(
							'name'           => 'Booking and Appointment',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/booking.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-booking/',
							'description'    => _x(
								'Enable a booking/appointment system to manage renting or booking of services, rooms, houses, cars, accommodation facilities and so on.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH Bookings',
								'ywpdp'
							),
							'is_active'      => defined( 'YITH_WCBK_PREMIUM' ),
							'is_recommended' => false,

						),
						'request-a-quote'        => array(
							'name'           => 'Request a Quote',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/request-a-quote.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-request-a-quote/',
							'description'    => _x(
								'Hide prices and/or the "Add to cart" button and let your customers request a custom quote for every product.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Request a Quote',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_YWRAQ_PREMIUM' ),
							'is_recommended' => false,
						),
						'product-addons'         => array(
							'name'           => 'Product Add-Ons & Extra Options',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/product-add-ons.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-product-add-ons/',
							'description'    => _x(
								'Add paid or free advanced options to your product pages using fields like radio buttons, checkboxes, drop-downs, custom text inputs, and more.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Product Add-Ons',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_WAPO_PREMIUM' ),
							'is_recommended' => false,
						),
						'gift-cards'             => array(
							'name'           => 'Gift Cards',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/gift-card.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-gift-cards/',
							'description'    => _x(
								'Sell gift cards to increase your store\'s revenue and win new customers.',
								'[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Gift Cards',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_YWGC_PREMIUM' ),
							'is_recommended' => false,
						),
						'customize-my-account'   => array(
							'name'           => 'Customize My Account Page',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/customize-myaccount-page.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-customize-my-account-page/',
							'description'    => _x( 'Customize the My Account page of your customers by creating custom sections with promotions and ad-hoc content based on your needs.', '[YOUR STORE TOOLS TAB] Description for plugin YITH WooCommerce Customize My Account', 'ywdpd' ),
							'is_active'      => defined( 'YITH_WCMAP_PREMIUM' ),
							'is_recommended' => false,
						),
						'recover-abandoned-cart' => array(
							'name'           => 'Recover Abandoned Cart',
							'icon_url'       => YITH_YWDPD_ASSETS_URL . '/images/plugins/recover-abandoned-cart.svg',
							'url'            => '//yithemes.com/themes/plugins/yith-woocommerce-recover-abandoned-cart/',
							'description'    => _x(
								'Contact users who have added products to the cart without completing the order and try to recover lost sales.',
								'[YOUR STORE TOOLS TAB] Description for plugin Recover Abandoned Cart',
								'ywdpd'
							),
							'is_active'      => defined( 'YITH_YWRAC_PREMIUM' ),
							'is_recommended' => false,
						),
					),
				),
			);

			// enable shop manager to set Dynamic Pricing Options.
			$enable_shop_manager = YITH_WC_Dynamic_Options::is_shop_manager_enabled();
			if ( $enable_shop_manager ) {
				add_filter(
					'option_page_capability_yit_' . $args['parent'] . '_options',
					array(
						$this,
						'change_capability',
					)
				);
				$args['capability'] = 'manage_woocommerce';
			}

			/* === Fixed: not updated theme  === */
			if ( ! class_exists( 'YIT_Plugin_Panel_WooCommerce' ) ) {
				require_once YITH_YWDPD_DIR . '/plugin-fw/lib/yit-plugin-panel-wc.php';
			}

			$this->panel = new YIT_Plugin_Panel_WooCommerce( $args );

		}


		/**
		 * Add the action links to plugin admin page
		 *
		 * @param array $links Links plugin array.
		 *
		 * @return mixed
		 * @use      plugin_action_links_{$plugin_file_name}
		 * @since    1.0
		 */
		public function action_links( $links ) {
			$links = yith_add_action_links( $links, $this->panel_page, true );

			return $links;
		}

		/**
		 * Plugin_row_meta.
		 *
		 * Add the action links to plugin admin page.
		 *
		 * @param array  $new_row_meta_args The new plugin meta.
		 * @param array  $plugin_meta       The plugin meta.
		 * @param string $plugin_file       The filename of plugin.
		 * @param array  $plugin_data       The plugin data.
		 * @param string $status            The plugin status.
		 * @param string $init_file         The filename of plugin.
		 *
		 * @return   array
		 * @since    1.0
		 * @use plugin_row_meta
		 */
		public function plugin_row_meta( $new_row_meta_args, $plugin_meta, $plugin_file, $plugin_data, $status, $init_file = 'YITH_YWDPD_INIT' ) {
			if ( defined( $init_file ) && constant( $init_file ) === $plugin_file ) {
				$new_row_meta_args['slug']       = YITH_YWDPD_SLUG;
				$new_row_meta_args['is_premium'] = true;
			}

			return $new_row_meta_args;
		}

		/**
		 * Register plugins for activation tab
		 *
		 * @return void
		 * @since    2.0.0
		 */
		public function register_plugin_for_activation() {
			if ( ! class_exists( 'YIT_Plugin_Licence' ) ) {
				require_once YITH_YWDPD_DIR . 'plugin-fw/licence/lib/yit-licence.php';
				require_once YITH_YWDPD_DIR . 'plugin-fw/licence/lib/yit-plugin-licence.php';
			}
			YIT_Plugin_Licence()->register( YITH_YWDPD_INIT, YITH_YWDPD_SECRET_KEY, YITH_YWDPD_SLUG );
		}

		/**
		 * Register plugins for update tab
		 *
		 * @return void
		 * @since    2.0.0
		 */
		public function register_plugin_for_updates() {
			if ( ! class_exists( 'YIT_Upgrade' ) ) {
				require_once YITH_YWDPD_DIR . 'plugin-fw/lib/yit-upgrade.php';
			}
			YIT_Upgrade()->register( YITH_YWDPD_SLUG, YITH_YWDPD_INIT );
		}

		/**
		 * Delete transient on post save or insert
		 *
		 * @param integer $post_id Post id.
		 * @param WP_Post $post    Post.
		 *
		 * @return void
		 * @since  1.0.0
		 */
		public function delete_transient( $post_id, $post ) {
			if ( 'ywdpd_discount' === $post->post_type ) {
				ywdpd_delete_plugin_transient();
				ywdpd_clear_modal_rule_cookie();
			}
		}

		/**
		 * Process some action after save a rule
		 *
		 * @param integer $post_id Post id.
		 * @param WP_Post $post    Post.
		 *
		 * @return void
		 * @throws Exception The exception.
		 */
		public function process_save_rule( $post_id, $post ) {
			$action = isset( $_REQUEST['action'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['action'] ) ) : false; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( 'ywdpd_discount' === $post->post_type && 'editpost' === $action ) {
				$this->store_schedule_meta( $post_id );
				$this->delete_transient( $post_id, $post );
				$this->add_rule_data_store( $post_id );
				ywdpd_clear_modal_rule_cookie();
				$this->redirect_to_list();
			}
		}

		/**
		 * Process the delete action
		 *
		 * @param int     $post_id The post id.
		 * @param WP_Post $post    The post.
		 *
		 * @return void
		 * @throws Exception The exception.
		 */
		public function process_delete_rule( $post_id, $post ) {
			if ( 'ywdpd_discount' === $post->post_type ) {
				$data_store = WC_Data_Store::load( 'ywdpd_dynamic_rules' );
				if ( $data_store->rule_exists( $post_id ) ) {
					$data_store->delete( $post_id );
				}
				$this->delete_transient( $post_id, $post );
			}
		}

		/**
		 * Add in the data store the rule
		 *
		 * @param int $post_id The rule id.
		 *
		 * @return void
		 * @throws Exception The error.
		 */
		public function add_rule_data_store( $post_id ) {
			$data_store = WC_Data_Store::load( 'ywdpd_dynamic_rules' );
			if ( $data_store->rule_exists( $post_id ) ) {
				$data_store->update( $post_id );
			} else {
				$data_store->create( $post_id );
			}
		}

		/**
		 * Add in the rule two meta for the schedule dates
		 * Is useful for meta query about scheduled rules
		 *
		 * @param int $post_id The post id.
		 *
		 * @return void
		 */
		public function store_schedule_meta( $post_id ) {
			if ( isset( $_REQUEST['yit_metaboxes']['_schedule_discount_mode'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$schedule_option = wp_unslash( $_REQUEST['yit_metaboxes']['_schedule_discount_mode'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				if ( 'schedule_dates' === $schedule_option['schedule_type'] ) {
					update_post_meta( $post_id, '_is_scheduled', 'yes' );
					update_post_meta( $post_id, '_schedule_from', $schedule_option['schedule_from'] );
					update_post_meta( $post_id, '_schedule_to', $schedule_option['schedule_to'] );

				} else {
					update_post_meta( $post_id, '_is_scheduled', 'no' );
					delete_post_meta( $post_id, '_schedule_from' );
					delete_post_meta( $post_id, '_schedule_to' );
				}
			}
		}

		/**
		 * Return to rule list
		 *
		 * @since 4.0.0
		 */
		public function redirect_to_list() {

			$url        = admin_url( 'edit.php' );
			$query_args = array(
				'post_type' => 'ywdpd_discount',
				'orderby'   => 'date_modified',
				'order'     => 'desc',
			);

			$url = esc_url_raw( add_query_arg( $query_args, $url ) );
			wp_safe_redirect( $url );
			exit;

		}

		/**
		 * Change messages when a post type is updated.
		 *
		 * @param array $messages Array of messages.
		 *
		 * @return array
		 * @since  1.3.0
		 */
		public function post_updated_messages( $messages ) {
			global $post;

			$messages['ywdpd_discount'] = array(
				0  => '', // Unused. Messages start at index 1.
				1  => __( 'Dynamic rule updated.', 'ywdpd' ),
				2  => __( 'Dynamic rule updated.', 'ywdpd' ),
				3  => __( 'Dynamic rule deleted.', 'ywdpd' ),
				4  => __( 'Dynamic rule updated.', 'ywdpd' ),
				5  => __( 'Dynamic rule restored.', 'ywdpd' ),
				6  => __( 'Dynamic rule updated.', 'ywdpd' ),
				7  => __( 'Dynamic rule saved.', 'ywdpd' ),
				8  => __( 'Dynamic rule submitted.', 'ywdpd' ),
				9  => sprintf(
				/* translators: %s: date */
					__( 'Dynamic rule scheduled for: %s.', 'yith-woocommerce-role-based-prices' ),
					'<strong>' . date_i18n( __( 'M j, Y @ G:i', 'ywdpd' ), strtotime( $post->post_date ) ) . '</strong>'
				),
				10 => __( 'Dynamic rule draft updated.', 'ywdpd' ),
				11 => __( 'Dynamic rule updated and sent.', 'ywdpd' ),
			);

			return $messages;
		}

		/**
		 * Specify custom bulk actions messages for different post types.
		 *
		 * @param array $bulk_messages Array of messages.
		 * @param array $bulk_counts   Array of how many objects were updated.
		 *
		 * @return array
		 * @since  1.3.0
		 */
		public function bulk_post_updated_messages( $bulk_messages, $bulk_counts ) {
			$bulk_messages['ywdpd_discount'] = array(
				/* translators: %s:  dynamic rule count */
				'updated'   => _n( '%s dynamic rule updated.', '%s dynamic rules updated.', $bulk_counts['updated'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'locked'    => _n( '%s dynamic rule not updated, somebody is editing it.', '%s dynamic rules not updated, somebody is editing them.', $bulk_counts['locked'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'deleted'   => _n( '%s dynamic rule permanently deleted.', '%s dynamic rules permanently deleted.', $bulk_counts['deleted'], 'ywdpd' ),
				/* translators: %s: dynamic rule count */
				'trashed'   => _n( '%s dynamic rule moved to the Trash.', '%s dynamic rules moved to the Trash.', $bulk_counts['trashed'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'untrashed' => _n( '%s dynamic rule restored from the Trash.', '%s dynamic rules restored from the Trash.', $bulk_counts['untrashed'], 'ywdpd' ),
			);

			return $bulk_messages;
		}

		/**
		 * Add the popup in the admin area
		 *
		 * @since  3.0.0
		 */
		public function show_admin_popup() {

			$first_check  = isset( $_GET['post_type'] ) && 'ywdpd_discount' === sanitize_text_field( wp_unslash( $_GET['post_type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$second_check = isset( $_GET['post'] ) && 'ywdpd_discount' === get_post_type( sanitize_text_field( wp_unslash( $_GET['post'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( $first_check || $second_check ) {
				require_once YITH_YWDPD_TEMPLATE_PATH . 'admin/dynamic-rules-popup.php';
			}
		}

		/**
		 * Create the default rule
		 *
		 * @since  3.0.0
		 */
		public function create_dynamic_rule() {

			if ( ! empty( $_GET['type'] ) ) {

				$type = sanitize_text_field( wp_unslash( $_GET['type'] ) );
				check_admin_referer( 'ywdpd-create-pop-rule-' . $type );
				$post_title = $this->generate_dynamic_rule_title( $type );
				$post_meta  = array(
					'_active'       => 'no',
					'_priority'     => ywdpd_get_last_priority() + 1,
					'_rule_name'    => $post_title,
					'_is_scheduled' => 'no',
					'_version'      => '3.0.0',
				);

				$new_post_id = wp_insert_post(
					array(
						'post_type'   => 'ywdpd_discount',
						'post_status' => 'publish',
						'post_title'  => $post_title,
					)
				);

				if ( 'three_x_two' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 3,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'two_x_one' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 2,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'bogo' === $type ) {
					$post_meta['_discount_mode'] = 'bogo';
					$post_meta['_rule_for']      = 'specific_products';
				} elseif ( '50%_off_2dn_unit' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 2,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 50,
					);
				} elseif ( 'black_friday' === $type ) {
					// Detect when will be the next Black friday!.
					$date_start                           = strtotime( 'fourth thursday of November' );
					$friday                               = strtotime( 'next friday', $date_start );
					$monday                               = strtotime( 'next monday', $friday );
					$post_meta['_discount_mode']          = 'discount_whole';
					$post_meta['_schedule_discount_mode'] = array(
						'schedule_type' => 'schedule_dates',
						'schedule_from' => gmdate( 'Y-m-d H:i', $friday ),
						'schedule_to'   => gmdate( 'Y-m-d H:i', $monday ),
					);
					$post_meta['_schedule_from']          = $post_meta['_schedule_discount_mode']['schedule_from'];
					$post_meta['_schedule_to']            = $post_meta['_schedule_discount_mode']['schedule_to'];
					$post_meta['_is_scheduled']           = 'yes';
				} elseif ( 'quantity_table' === $type ) {
					$post_meta['_discount_mode'] = 'bulk';
					$post_meta['_qty_type']      = 'range';
				} elseif ( 'user' === $type ) {
					$post_meta['_discount_mode']         = 'discount_whole';
					$post_meta['_user_rules']            = 'role_list';
					$post_meta['_simple_whole_discount'] = array(
						'discount_mode'  => 'percentage',
						'discount_value' => 25,
					);
				} elseif ( 'buy_x_get_y' === $type ) {
					$post_meta['_discount_mode']                      = 'special_offer';
					$post_meta['_rule_for']                           = 'specific_products';
					$post_meta['_active_apply_discount_to']           = 'yes';
					$post_meta['_rule_apply_adjustment_discount_for'] = 'specific_products';
					$post_meta['so-rule']                             = array(
						'purchase'        => 1,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'gift_rule' === $type ) {
					$post_meta['_discount_mode'] = 'gift_products';
				} elseif ( 'cart_rule' === $type ) {
					$post_meta['_discount_mode']       = 'cart';
					$post_meta['_apply_discount_mode'] = 'always';
					$post_meta['_apply_discount']      = 'yes';
					$post_meta['_discount_rule']       = array(
						'discount_type'   => 'percentage',
						'discount_amount' => 20,
					);
				} elseif ( 'last_deals' === $type ) {
					$post_meta['_discount_mode'] = 'last_deals';
				} else {
					$post_meta['_discount_mode']       = 'cart';
					$post_meta['_allow_free_shipping'] = 'yes';
				}

				foreach ( $post_meta as $meta_key => $meta_value ) {
					update_post_meta( $new_post_id, $meta_key, $meta_value );
				}

				$url = add_query_arg(
					array(
						'post'   => $new_post_id,
						'action' => 'edit',
					),
					admin_url( 'post.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;
			}
		}

		/**
		 * Create an empty dynamic rule
		 *
		 * @since  3.0.0
		 */
		public function create_simple_dynamic_rule() {
			if ( ! empty( $_GET['type'] ) ) {
				$type = sanitize_text_field( wp_unslash( $_GET['type'] ) );
				check_admin_referer( 'ywdpd-create-pop-simple-rule-' . $type );
				$post_title = $this->generate_dynamic_rule_title( $type );
				$post_meta  = array(
					'_active'    => 'no',
					'_priority'  => ywdpd_get_last_priority() + 1,
					'_rule_name' => $post_title,
					'_version'   => '3.0.0',
				);

				$new_post_id = wp_insert_post(
					array(
						'post_type'   => 'ywdpd_discount',
						'post_status' => 'publish',
						'post_title'  => $this->generate_dynamic_rule_title( $type ),
					)
				);

				if ( 'simple_qty' === $type ) {
					$post_meta['_discount_mode'] = 'bulk';
				} elseif ( 'simple_special_offer' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
				} elseif ( 'simple_gift_products' === $type ) {
					$post_meta['_discount_mode'] = 'gift_products';
				} elseif ( 'simple_whole_shop' === $type ) {
					$post_meta['_discount_mode'] = 'discount_whole';
				} elseif ( 'simple_category_discount' === $type ) {
					$post_meta['_discount_mode'] = 'category_discount';
				} else {
					$post_meta['_discount_mode'] = 'cart';
				}

				foreach ( $post_meta as $meta_key => $meta_value ) {
					update_post_meta( $new_post_id, $meta_key, $meta_value );
				}

				$url = add_query_arg(
					array(
						'post'   => $new_post_id,
						'action' => 'edit',
					),
					admin_url( 'post.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;

			}
		}

		/**
		 * Return the unique rule title
		 *
		 * @param string $type The rule type.
		 *
		 * @return string
		 * @since  3.0.0
		 */
		public function generate_dynamic_rule_title( $type ) {

			$post_title  = str_replace( '_', ' ', $type );
			$check_title = $post_title;
			$counter     = 1;

			while ( post_exists( $check_title, '', '', 'ywdpd_discount' ) ) {
				$check_title = $post_title . ' ' . $counter;
				$counter ++;
			}

			return $check_title;
		}

		/**
		 * Add categories exclusion table.
		 *
		 * @access public
		 * @since  3.0.0
		 */
		public function show_the_exclusion_table() {

			$get           = $_GET; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$template_file = YITH_YWDPD_VIEW_PATH . '/panel/custom-tabs/ywdpd-exclusions-table.php';

			if ( isset( $get['page'] ) && 'yith_woocommerce_dynamic_pricing_and_discounts' === $get['page'] && isset( $get['tab'] ) && 'exclusions' === $get['tab'] && file_exists( $template_file ) ) {

				$exclusions_prod = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-prod-list' ) ) );
				$exclusions_cat  = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-cat-list' ) ) );
				$exclusions_tag  = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-tag-list' ) ) );
				$list            = array_merge( $exclusions_prod, $exclusions_cat, $exclusions_tag );

				$is_blank = ! YWDPD_Exclusion_List_Data_Store::has_items();

				wp_enqueue_style( 'ywdpd_exclusion' );
				wp_enqueue_script( 'ywdpd_exclusion' );
				include_once YITH_YWDPD_INC . '/admin/class-ywdpd-exclusions-list-table.php';

				$table = new YWDPD_Exclusions_List_Table();
				$table->prepare_items();

				include_once $template_file;
			}
		}

		/**
		 * Add items in the exclusion table
		 *
		 * @since  3.0.0
		 */
		public function add_items_in_exclusion_list() {
			$posted = $_POST; //phpcs:ignore WordPress.Security.NonceVerification.Missing
			if ( ! isset( $posted['_nonce'], $posted['ywdpd-exclusion-type'] ) || ! wp_verify_nonce( $posted['_nonce'], 'yith_ywdpd_add_exclusions' ) ) {
				return;
			}

			$term_type = $posted['ywdpd-exclusion-type'];
			$term_ids  = $posted[ 'ywdpd_add_' . $term_type ];

			foreach ( $term_ids as $term_id ) {
				YWDPD_Exclusion_List_Data_Store::insert( $term_id, $term_type );
			}
		}

		/**
		 * Delete item from the exclusion list
		 *
		 * @since  3.0.0
		 */
		public function delete_item_from_exclusion_list() {

			if ( isset( $_GET['term_id'] ) && ( isset( $_GET['action'] ) && 'delete_from_exclusion' === $_GET['action'] ) ) {
				$term_id = sanitize_text_field( wp_unslash( $_GET['term_id'] ) );
				check_admin_referer( 'ywdpd-delete-exclusion_' . $term_id );
				YWDPD_Exclusion_List_Data_Store::delete( $term_id );

				$url = add_query_arg(
					array(
						'page' => 'yith_woocommerce_dynamic_pricing_and_discounts',
						'tab'  => 'exclusions',
					),
					admin_url( 'admin.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;
			}
		}

		/**
		 * Show the dynamic rule in the item
		 *
		 * @param int                   $item_id The item id.
		 * @param WC_Order_Item_Product $item    The item.
		 * @param WC_Product            $product The product.
		 *
		 * @since  3.0.0
		 */
		public function show_custom_meta_item_in_order( $item_id, $item, $product = false ) {
			if ( $item instanceof WC_Order_Item_Product ) {
				$dynamic_rules = $item->get_meta( '_ywdpd_discounts' );
				if ( ! empty( $dynamic_rules ) ) {
					$custom_meta = '';

					foreach ( $dynamic_rules['applied_discounts'] as $applied_discount ) {
						if ( isset( $applied_discount['set_id'] ) ) {
							$rule_id = $applied_discount['set_id'];
							$rule    = ywdpd_get_rule( $rule_id );
						} else {
							$rule = $applied_discount['by'];
						}
						if ( $rule instanceof YWDPD_Price_Rule && ! empty( $rule->get_name() ) ) {
							$custom_meta .= '<li>' . $rule->get_name() . '</li>';
						}
					}
					if ( ! empty( $custom_meta ) ) { ?>
                        <div class="view">
                            <table cellspacing="0" class="display_meta">
                                <th><?php esc_html_e( 'Offer applied:', 'ywdpd' ); ?></th>
                                <td>
                                    <ul style="margin:0;"><?php echo wp_kses_post( $custom_meta ); ?>
                                    </ul>
                                </td>
                            </table>
                        </div>
						<?php
					}
				}
			}
		}

		/**
		 * Show the right coupon name in backend
		 *
		 * @param string               $code        The coupon code.
		 * @param WC_Order_Item_Coupon $coupon_item The coupon item object.
		 *
		 * @since  3.4.0
		 */
		public function show_right_coupon_info( $code, $coupon_item ) {
			$new_code = $coupon_item->get_meta( '_ywdpd_coupon_info' );

			if ( ! empty( $new_code ) ) {
				$code = $new_code;
			}

			return $code;
		}

		/**
		 * Show the rule info under the rule name
		 *
		 * @return void
		 * @since 4.0
		 */
		public function add_rule_info_badge() {
			$screen    = function_exists( 'get_current_screen' ) ? get_current_screen() : false;
			$screen_id = $screen ? $screen->id : false;

			if ( $screen_id === $this->post_type_name ) {
				$post_id = isset( $_GET['post'] ) ? sanitize_text_field( wp_unslash( $_GET['post'] ) ) : false; // phpcs:ignore WordPress.Security.NonceVerification

				$discount_mode = get_post_meta( $post_id, '_discount_mode', true );
				$rule_type     = ywdpd_get_price_rule_type( $discount_mode );

				?>
                <div id="ywdpd-info-wrapper">
                    <span class="ywdpd-info-label"><?php esc_html_e( 'Type:', 'ywdpd' ); ?></span>
                    <span class="ywdpd-info-content"><?php echo esc_html( $rule_type ); ?></span>
                </div>
                <script type="text/javascript">
                    (function () {
                        var wrap = document.querySelector('.wrap'),
                            backTo = document.querySelector('#ywdpd-info-wrapper');

                        wrap.insertBefore(backTo, wrap.childNodes[3]);
                    })();
                </script>
				<?php
			}

		}

		/**
		 * Print save button in edit page.
		 *
		 * @param WP_Post $post The post.
		 */
		public function print_save_button_in_edit_page( $post ) {
			if ( ! ! $post && isset( $post->post_type ) && $post->post_type === $this->post_type_name ) {
				global $post_id;
				$is_updating      = ! ! $post_id;
				$save_text        = __( 'Save', 'ywdpd' );
				$post_type_object = get_post_type_object( $this->post_type_name );
				$single           = $post_type_object->labels->singular_name ?? '';

				if ( $single ) {

					$save_text = __( 'Save rule', 'ywdpd' );
				}
				?>
                <div class="ywdpd-post-type__actions yith-plugin-ui">
					<?php if ( $is_updating ) : ?>
                        <button id="ywdpd-post-type__save"
                                class="yith-plugin-fw__button--primary yith-plugin-fw__button--xl"><?php echo esc_html( $save_text ); ?></button>
					<?php else : ?>
                        <input id=ywdpd-post-type__save" type="submit"
                               class="yith-plugin-fw__button--primary yith-plugin-fw__button--xl" name="publish"
                               value="<?php echo esc_html( $save_text ); ?>">
					<?php endif; ?>

                    <a id="ywdpd-post-type__float-save"
                       class="yith-plugin-fw__button--primary yith-plugin-fw__button--xl yith-plugin-fw-animate__appear-from-bottom"><?php echo esc_html( $save_text ); ?></a>
                </div>
				<?php
			}
		}

		/**
		 * Show the notice in the plugin panel
		 *
		 * @return void
		 */
		public function print_notices() {
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ( isset( $_GET['post_type'] ) && 'ywdpd_discount' === $_GET['post_type'] ) || ( isset( $_GET['page'] ) && 'yith_woocommerce_dynamic_pricing_and_discounts' === $_GET['page'] ) ) {
				if ( ywdpd_sync_dynamic_rules_batch() ) {
					yith_ywdpd_print_notice( __( 'We are running an update in the db. This could take some minutes', 'ywdpd' ), 'info', true, 'dynamic-update' );
				}
			}
		}
		/**
		 * If a cart rule is updated clear the shipping transient
		 *
		 * @param int $post_id The post id.
		 *
		 * @return void
		 * @since 4.0
		 */
		public function invalidate_shipping_cache( $post_id ) {
			$cart_rule = ywdpd_get_rule( $post_id );
			if ( 'cart' === $cart_rule->get_type() ) {
				WC_Cache_Helper::get_transient_version( 'shipping', true );
			}
		}
	}
}

/**
 * Unique access to instance of YITH_WC_Dynamic_Pricing_Admin class
 *
 * @return YITH_WC_Dynamic_Pricing_Admin
 */
function YITH_WC_Dynamic_Pricing_Admin() {  //phpcs:ignore

	return YITH_WC_Dynamic_Pricing_Admin::get_instance();
}

YITH_WC_Dynamic_Pricing_Admin();
