<?php // phpcs:ignore WordPress.Files.FileName.InvalidClassFileName
/**
 * Abstract Render Block Class.
 *
 * @package YITH\DynamicPricingAndDiscounts\Classes
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'YWDPD_Render_Block' ) ) {
	/**
	 * Abstract Render Block Class.
	 *
	 * @since 4.0.0
	 */
	abstract class YWDPD_Render_Block {

		/**
		 * Block attributes
		 *
		 * @var array
		 */
		protected $attributes = array();

		/**
		 * Object data.
		 *
		 * @var array
		 */
		protected $data = array(
			'allow_blank_state' => 'no',
			'is_preview'        => 'no',
		);

		/**
		 * YWDPD_Render_Block constructor.
		 *
		 * @param array $attributes Block attributes.
		 */
		public function __construct( $attributes = array() ) {
			$this->attributes = $this->parse_attributes( $attributes );
		}

		/**
		 * Parse attributes.
		 *
		 * @param array $attributes Attributes.
		 *
		 * @return array
		 */
		protected function parse_attributes( $attributes ) {
			return wp_parse_args( $attributes, $this->attributes );
		}

		/**
		 * Get allow_blank_state value.
		 *
		 * @return string
		 */
		public function get_allow_blank_state() {
			return $this->data['allow_blank_state'];
		}

		/**
		 * Get is_preview value
		 *
		 * @return string
		 */
		public function get_is_preview() {
			return $this->data['is_preview'];
		}

		/**
		 * Is the empty state allowed?
		 *
		 * @return bool
		 */
		public function is_blank_state_allowed() {
			return 'yes' === $this->get_allow_blank_state();
		}

		/**
		 * Is preview ?
		 *
		 * @return bool
		 */
		public function is_preview() {
			return 'yes' === $this->get_is_preview();
		}

		/**
		 * Set the allow_blank_state value.
		 *
		 * @param bool|string $value The value to be set.
		 */
		public function set_allow_blank_state( $value ) {
			$this->data['allow_blank_state'] = wc_bool_to_string( $value );
		}

		/**
		 * Set is_preview value
		 *
		 * @param bool|string $value The value.
		 */
		public function set_is_preview( $value ) {
			$this->data['is_preview'] = wc_bool_to_string( $value );
		}

		/**
		 * Render
		 */
		abstract public function render();

		/**
		 * Retrieve blank state params.
		 *
		 * @return array
		 */
		public function get_blank_state_params() {
			return array();
		}

		/**
		 * Render an empty state.
		 */
		public function render_blank_state() {
			if ( ! wp_style_is( 'yith-plugin-ui', 'registered' ) ) {
				$plugin_fw_assets = class_exists( 'YIT_Assets' ) && is_callable( 'YIT_Assets::instance' ) ? YIT_Assets::instance() : false;

				if ( $plugin_fw_assets && is_callable( array( $plugin_fw_assets, 'register_styles_and_scripts' ) ) ) {
					$plugin_fw_assets->register_styles_and_scripts();
				}
			}

			wp_enqueue_style( 'yith-plugin-ui' );

			$blank_state_params         = $this->get_blank_state_params();
			$blank_state_params['type'] = 'list-table-blank-state';

			yith_plugin_fw_get_component( $blank_state_params, true );
		}

		/**
		 * Is this a block preview request?
		 */
		protected function is_block_preview() {
			return defined( 'YWDPD_BLOCK_PREVIEW' ) && YWDPD_BLOCK_PREVIEW;
		}
	}
}
