<?php
/**
 * This file contain all functions with the common options
 *
 * @since   4.0
 * @package YITH\DynamicPricingAndDiscounts\Functions
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
if ( ! function_exists( 'ywdpd_get_extra_desc_for_wpml' ) ) {
	/**
	 * Return an extra description if WPML is active
	 *
	 * @return string
	 */
	function ywdpd_get_extra_desc_for_wpml() {
		$extra_desc_for_wpml = '';
		global $sitepress;
		if ( ! is_null( $sitepress ) ) {
			// @since 3.0.0
			$extra_desc_for_wpml = sprintf( '<p>%s</p>', __( 'If you want to show this text in a different language, please, type your text inside a tag like this: <code>[en]text here[/en]</code> for English, <code>[it]text here[/it]</code> for Italian, and so on', 'ywdpd' ) );
		}

		return $extra_desc_for_wpml;
	}
}
if ( ! function_exists( 'ywdpd_get_apply_to_options' ) ) {

	/**
	 * Return the "Apply to" configuration by rule type
	 *
	 * @param string $rule_type    The rule type.
	 * @param bool   $only_exclude Incluor or not the exclude options.
	 *
	 * @return array
	 */
	function ywdpd_get_apply_to_options( $rule_type, $only_exclude = false ) {

		switch ( $rule_type ) {

			case 'special_offer':
				$label = __( 'Create a special offer rule for the purchase of:', 'ywdpd' );
				break;
			case 'bogo':
				$label = __( 'Create a BOGO rule for the purchase of:', 'ywdpd' );
				break;
			case 'gift_products':
				$label = __( 'Offer gifts if user buys:', 'ywdpd' );
				break;
			default:
				$label = __( 'Create a quantity rule for the purchase of:', 'ywdpd' );
				break;
		}

		/**
		 * APPLY_FILTERS: ywdpd_get_rule_for_options
		 *
		 * Allow to change the options for the field "Rule for"
		 *
		 * @param array $rule_for_list The options
		 *
		 * @return array
		 */
		$rule_for_list = apply_filters(
			'ywdpd_get_rule_for_options',
			array(
				'all_products'        => __( 'All products', 'ywdpd' ),
				'specific_products'   => __( 'Specific products', 'ywdpd' ),
				'specific_categories' => __( 'Specific product categories', 'ywdpd' ),
				'specific_tag'        => __( 'Specific product tags', 'ywdpd' ),
			)
		);

		/**
		 * APPLY_FILTERS: ywdpd_get_rule_for_search_fields
		 *
		 * Allow to change the search field options for the field "Rule for"
		 *
		 * @param array $rule_for_search_fields The options
		 *
		 * @return array
		 */
		$rule_for_search_fields = apply_filters(
			'ywdpd_get_rule_for_search_fields',
			array(
				'rule_for_products_list'   => array(
					'label'    => __( 'Apply rule to:', 'ywdpd' ),
					'type'     => 'ajax-products',
					'desc'     => __( 'Search the product(s) to include in the rule', 'ywdpd' ),
					'multiple' => true,
					'std'      => array(),
					'data'     => array(
						'action'     => 'woocommerce_json_search_products_and_variations',
						'security'   => wp_create_nonce( 'search-products' ),
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_rule_for',
									'value' => 'specific_products',
								),
							)
						),
					),
					'required' => true,
				),
				'rule_for_categories_list' => array(
					'label'    => __( 'Apply rule to:', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_cat',
						'placeholder' => __( 'Search for a category', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_for',
									'value' => 'specific_categories',
								),
							)
						),
					),
					'multiple' => true,
					'desc'     => __( 'Search the product categories to include in the rule', 'ywdpd' ),
					'required' => true,
				),
				'rule_for_tags_list'       => array(
					'label'    => __( 'Apply rule to:', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_tag',
						'placeholder' => __( 'Search for a tag', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_for',
									'value' => 'specific_tag',
								),
							)
						),
					),
					'desc'     => __( 'Search the product tags to include in the rule', 'ywdpd' ),
					'multiple' => true,
					'required' => true,
				),
			)
		);
		$include                = array_merge(
			array(
				'rule_for' => array(
					'label'   => $label,
					'desc'    => __( 'Choose if you want to create this rule for all products or for specific products, categories, or tags', 'ywdpd' ),
					'type'    => 'select',
					'class'   => 'wc-enhanced-select',
					'options' => $rule_for_list,
					'std'     => 'all_products',
				),
			),
			$rule_for_search_fields,
		);

		/**
		 * APPLY_FILTERS: ywdpd_get_exclude_rule_for_options
		 *
		 * Allow to change the options for the field "Exclude Rule for"
		 *
		 * @param array $exclude_rule_for_options The options
		 *
		 * @return array
		 */
		$exclude_rule_for_options = apply_filters(
			'ywdpd_get_exclude_rule_for_options',
			array(
				'specific_products'   => __( 'Specific products', 'ywdpd' ),
				'specific_categories' => __( 'Specific product categories', 'ywdpd' ),
				'specific_tag'        => __( 'Specific product tags', 'ywdpd' ),
			)
		);

		/**
		 * APPLY_FILTERS: ywdpd_get_exclude_rule_for_search_fields
		 *
		 * Allow to change the search field options for the field "Exclude Rule for"
		 *
		 * @param array $exclude_rule_for_search_fields The options
		 *
		 * @return array
		 */
		$exclude_rule_for_search_fields = apply_filters(
			'ywdpd_get_exclude_rule_for_search_fields',
			array(
				'exclude_rule_for_products_list'   => array(
					'label'    => __( 'Choose which product(s) to exclude', 'ywdpd' ),
					'type'     => 'ajax-products',
					'desc'     => __( 'Search the product(s) to exclude from this rule', 'ywdpd' ),
					'multiple' => true,
					'std'      => array(),
					'data'     => array(
						'action'     => 'woocommerce_json_search_products_and_variations',
						'security'   => wp_create_nonce( 'search-products' ),
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_exclude_rule_for',
									'value' => 'specific_products',
								),
							)
						),
					),
					'required' => true,
				),
				'exclude_rule_for_categories_list' => array(
					'label'    => __( 'Choose the product categories to exclude', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_cat',
						'placeholder' => __( 'Search for a category', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_exclude_rule_for',
									'value' => 'specific_categories',
								),
							)
						),
					),
					'multiple' => true,
					'desc'     => __( 'Search the product categories to exclude from this rule', 'ywdpd' ),
					'required' => true,
				),
				'exclude_rule_for_tags_list'       => array(
					'label'    => __( 'Choose which product tags to exclude', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_tag',
						'placeholder' => __( 'Search for a tag', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_exclude_rule_for',
									'value' => 'specific_tag',
								),
							)
						),
					),
					'desc'     => __( 'Search the product tags to exclude from this rule', 'ywdpd' ),
					'multiple' => true,
					'required' => true,
				),
			)
		);
		$exclude                        = array_merge(
			array(
				'active_exclude'   => array(
					'label' => __( 'Exclude products from this rule', 'ywdpd' ),
					'desc'  => __( 'Enable if you want to exclude specific products from this rule', 'ywdpd' ),
					'type'  => 'onoff',
					'std'   => 'no',
					'data'  => array(
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
							)
						),
					),
				),
				'exclude_rule_for' => array(
					'label'   => __( 'Exclude', 'ywdpd' ),
					'desc'    => __( 'Choose if you want to exclude some specific products, categories, or tags from this rule', 'ywdpd' ),
					'type'    => 'select',
					'class'   => 'wc-enhanced-select',
					'options' => $exclude_rule_for_options,
					'std'     => 'specific_products',
					'data'    => array(
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
							)
						),
					),
				),
			),
			$exclude_rule_for_search_fields
		);

		return $only_exclude ? $exclude : array_merge( $include, $exclude );
	}
}

if ( ! function_exists( 'ywdpd_get_adjustment_to_options' ) ) {
	/**
	 * Return the "Apply adjustment to" configuration by rule type
	 *
	 * @param string $rule_type The rule type.
	 *
	 * @return array
	 */
	function ywdpd_get_adjustment_to_options( $rule_type ) {

		/**
		 * APPLY_FILTERS: ywdpd_get_adjustment_to_options
		 *
		 * Allow to change the options for the field "Adjustment to"
		 *
		 * @param array $adjustment_to_list The options
		 *
		 * @return array
		 */
		$adjustment_to_options = apply_filters(
			'ywdpd_get_adjustment_to_options',
			array(
				'all_products'        => __( 'All products', 'ywdpd' ),
				'specific_products'   => __( 'Specific products', 'ywdpd' ),
				'specific_categories' => __( 'Specific product categories', 'ywdpd' ),
				'specific_tag'        => __( 'Specific product tags', 'ywdpd' ),
			)
		);
		/**
		 * APPLY_FILTERS: ywdpd_get_adjustment_to_search_fields
		 *
		 * Allow to change the options for the field "Adjustment to"
		 *
		 * @param array $adjustment_to_search_fields The options
		 *
		 * @return array
		 */
		$adjustment_to_search_fields = apply_filters(
			'ywdpd_get_adjustment_to_search_fields',
			array(
				'apply_adjustment_products_list'   => array(
					'label'    => __( 'Choose which product(s) to include', 'ywdpd' ),
					'type'     => 'ajax-products',
					'desc'     => __( 'Search the product(s) to include in this discount', 'ywdpd' ),
					'multiple' => true,
					'data'     => array(
						'action'     => 'woocommerce_json_search_products_and_variations',
						'security'   => wp_create_nonce( 'search-products' ),
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_rule_apply_adjustment_discount_for',
									'value' => 'specific_products',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'required' => true,
				),
				'apply_adjustment_categories_list' => array(
					'label'    => __( 'Choose which product categories to include', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_cat',
						'placeholder' => __( 'Search for a category', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_apply_adjustment_discount_for',
									'value' => 'specific_categories',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'multiple' => true,
					'desc'     => __( 'Search the product categories to include in the discount', 'ywdpd' ),
					'required' => true,
				),
				'apply_adjustment_tags_list'       => array(
					'label'    => __( 'Choose which product tags to include', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_tag',
						'placeholder' => __( 'Search for a tag', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_apply_adjustment_discount_for',
									'value' => 'specific_tag',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'desc'     => __( 'Search the product tags to include in the discount', 'ywdpd' ),
					'multiple' => true,
					'required' => true,
				),
			)
		);

		/**
		 * APPLY_FILTERS: ywdpd_get_exclude_adjustment_to_options
		 *
		 * Allow to change the options for the field "Exclude Adjustment to"
		 *
		 * @param array $exclude_adjustment_to_options The options
		 *
		 * @return array
		 */
		$exclude_adjustment_to_options = apply_filters(
			'ywdpd_get_exclude_adjustment_to_options',
			array(
				'specific_products'   => __( 'Specific products', 'ywdpd' ),
				'specific_categories' => __( 'Specific product categories', 'ywdpd' ),
				'specific_tag'        => __( 'Specific product tags', 'ywdpd' ),
			)
		);

		/**
		 * APPLY_FILTERS: ywdpd_get_exclude_adjustment_to_search_fields
		 *
		 * Allow to change the options for the field "Exclude Adjustment to"
		 *
		 * @param array $exclude_adjustment_to_search_fields The options
		 *
		 * @return array
		 */
		$exclude_adjustment_to_search_fields = apply_filters(
			'ywdpd_get_exclude_adjustment_to_search_fields',
			array(
				'apply_adjustment_products_list_excluded'   => array(
					'label'    => __( 'Choose which product(s) to exclude', 'ywdpd' ),
					'type'     => 'ajax-products',
					'desc'     => __( 'Search the product(s) to exclude', 'ywdpd' ),
					'multiple' => true,
					'data'     => array(
						'action'     => 'woocommerce_json_search_products_and_variations',
						'security'   => wp_create_nonce( 'search-products' ),
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_exclude_apply_adjustment_rule_for',
									'value' => 'specific_products',
								),
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'required' => true,
				),
				'apply_adjustment_categories_list_excluded' => array(
					'label'    => __( 'Choose the product categories to exclude', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_cat',
						'placeholder' => __( 'Search for a category', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_exclude_apply_adjustment_rule_for',
									'value' => 'specific_categories',
								),
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'multiple' => true,
					'desc'     => __( 'Search the product categories to exclude', 'ywdpd' ),
					'required' => true,
				),
				'apply_adjustment_tags_list_excluded'       => array(
					'label'    => __( 'Choose which product tags to exclude', 'ywdpd' ),
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => 'product_tag',
						'placeholder' => __( 'Search for a tag', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_exclude_apply_adjustment_rule_for',
									'value' => 'specific_tag',
								),
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
					'desc'     => __( 'Search the product tags to exclude', 'ywdpd' ),
					'multiple' => true,
					'required' => true,
				),
			)
		);

		return array_merge(
			array(
				'active_apply_discount_to'           => array(
					'label' => __( 'Apply discount to a different product or category', 'ywdpd' ),
					'desc'  => __( 'Enable if you want to apply the discount to other products rather than the products selected for this quantity rule. For example: if a customer purchases 100 business cards (product A), they can get a 10% discount on flyers (product B)', 'ywdpd' ),
					'type'  => 'onoff',
					'std'   => 'no',
				),
				'rule_apply_adjustment_discount_for' => array(
					'label'   => __( 'Apply discount to', 'ywdpd' ),
					'desc'    => __( 'Choose whether to apply the discount to all products or only to specific products, categories, or tags', 'ywdpd' ),
					'type'    => 'select',
					'class'   => 'wc-enhanced-select',
					'options' => $adjustment_to_options,
					'std'     => 'all_products',
					'data'    => array(
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
				),
			),
			$adjustment_to_search_fields,
			array(
				'active_apply_adjustment_to_exclude' => array(
					'label' => __( 'Exclude products from this discount', 'ywdpd' ),
					'desc'  => __( 'Enable if you want to exclude specific products from this discount', 'ywdpd' ),
					'type'  => 'onoff',
					'std'   => 'no',
					'data'  => array(
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
				),
				// EXCLUDE PRODUCT FOR APPLY ADJUSTMENT TO.
				'exclude_apply_adjustment_rule_for'  => array(
					'label'   => __( 'Exclude', 'ywdpd' ),
					'desc'    => __( 'Choose if you want to exclude some specific products, categories, or tags from this rule', 'ywdpd' ),
					'type'    => 'select',
					'class'   => 'wc-enhanced-select',
					'options' => $exclude_adjustment_to_options,
					'std'     => 'specific_products',
					'data'    => array(
						'ywdpd-deps' => wp_json_encode(
							array(
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
							)
						),
					),
				),
			),
			$exclude_adjustment_to_search_fields
		);
	}
}

if ( ! function_exists( 'ywdpd_get_user_options' ) ) {

	/**
	 * Return the "User rule" configuration by rule type
	 *
	 * @param string $rule_type The rule type.
	 *
	 * @return array
	 */
	function ywdpd_get_user_options( $rule_type ) {

		$exclude_label = __( 'Exclude users from this discount', 'ywdpd' );
		$exclude_desc  = __( 'Enable if you want to exclude specific users from this discount', 'ywdpd' );
		switch ( $rule_type ) {
			case 'special_offer':
				$label = __( 'Apply special offer to:', 'ywdpd' );
				break;
			case 'bogo':
				$label = __( 'Apply BOGO rule to:', 'ywdpd' );
				break;
			case 'gift_products':
				$label         = __( 'Offer gift products to:', 'ywdpd' );
				$exclude_label = __( 'Exclude users from this rule', 'ywdpd' );
				$exclude_desc  = __( 'Enable if you want to exclude specific users from this rule', 'ywdpd' );
				break;
			case 'last_deals':
				$label = __( 'Offer deal to:', 'ywdpd' );
				break;
			default:
				$label = __( 'Apply discount to:', 'ywdpd' );
				break;
		}

		/**
		 * APPLY_FILTERS: ywdpd_membership_plan_included_field
		 *
		 * Allow change the membership options.
		 *
		 * @param array $membership_plan_included The options
		 *
		 * @return array
		 */
		$membership_plan_included = apply_filters( 'ywdpd_membership_plan_included_field', '' );
		/**
		 * APPLY_FILTERS: ywdpd_membership_plan_excluded_field
		 *
		 * Allow change the membership options.
		 *
		 * @param array $membership_plan_excluded The options
		 *
		 * @return array
		 */
		$membership_plan_excluded = apply_filters( 'ywdpd_membership_plan_excluded_field', '' );

		return array(
			'user_rules'                           => array(
				'label'   => $label,
				'desc'    => __( 'Choose to apply the rule to all users or only to specific users or user roles', 'ywdpd' ),
				'type'    => 'select',
				'class'   => 'wc-enhanced-select',
				'options' => apply_filters(
					'ywdpdp_price_rule_user_options',
					array(
						'everyone'       => __( 'All users', 'ywdpd' ),
						'customers_list' => __( 'Only to specific users', 'ywdpd' ),
						'role_list'      => __( 'Only to specific user roles', 'ywdpd' ),
					)
				),
				'std'     => 'everyone',
			),
			'user_rules_role_list'                 => array(
				'label'    => __( 'User roles included', 'ywdpd' ),
				'desc'     => __( 'Search for the user roles you want to include in this rule', 'ywdpd' ),
				'type'     => 'select',
				'class'    => 'wc-enhanced-select',
				'multiple' => true,
				'options'  => ywdpd_get_roles(),
				'std'      => array(),
				'data'     => array(
					'placeholder' => __( 'Select user roles', 'ywdpd' ),
					'ywdpd-deps'  => wp_json_encode(
						array(
							array(
								'id'    => '_user_rules',
								'value' => 'role_list',
							),
						)
					),
				),
				'required' => true,
			),
			'user_rules_customers_list'            => array(
				'label'       => __( 'Users included', 'ywdpd' ),
				'type'        => 'custom',
				'action'      => 'ywdpd_show_custom_field',
				'ywdpd_type'  => 'search-customers',
				'desc'        => __( 'Search for the users you want to include in this rule', 'ywdpd' ),
				'required'    => true,
				'multiple'    => true,
				'allow_clear' => true,
				'data'        => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_user_rules',
								'value' => 'customers_list',
							),
						)
					),
				),
			),
			'user_rules_memberships_list'          => $membership_plan_included,
			'enable_user_rule_exclude'             => array(
				'label' => $exclude_label,
				'desc'  => $exclude_desc,
				'type'  => 'onoff',
				'std'   => 'no',
				'data'  => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'      => '_user_rules',
								'value'   => 'customers_list',
								'compare' => '!=',
							),
						)
					),
				),
			),
			'user_rule_exclude'                    => array(
				'label'   => __( 'Exclude', 'ywdpd' ),
				'desc'    => __( 'Choose if you want to exclude from this rule certain users or user roles', 'ywdpd' ),
				'type'    => 'select',
				'class'   => 'wc-enhanced-select',
				'options' => apply_filters(
					'ywdpdp_price_rule_user_exclude_options',
					array(
						'specific_customers' => __( 'Specific users', 'ywdpd' ),
						'specific_roles'     => __( 'Specific user roles', 'ywdpd' ),
					)
				),
				'std'     => 'specific_customers',
				'data'    => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_enable_user_rule_exclude',
								'value' => 'yes',
							),
							array(
								'id'      => '_user_rules',
								'value'   => 'customers_list',
								'compare' => '!=',
							),
						)
					),
				),
			),
			'user_rules_role_list_excluded'        => array(
				'label'       => __( 'Choose which user role(s) to exclude', 'ywdpd' ),
				'desc'        => __( 'Search for the user roles you want to exclude from this rule', 'ywdpd' ),
				'placeholder' => __( 'Select user roles', 'ywdpd' ),
				'type'        => 'select',
				'class'       => 'wc-enhanced-select',
				'multiple'    => true,
				'options'     => ywdpd_get_roles(),
				'std'         => array(),
				'required'    => true,
				'data'        => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_user_rule_exclude',
								'value' => 'specific_roles',
							),
							array(
								'id'    => '_enable_user_rule_exclude',
								'value' => 'yes',
							),
							array(
								'id'      => '_user_rules',
								'value'   => 'customers_list',
								'compare' => '!=',
							),
						)
					),
				),
			),
			'user_rules_customers_list_excluded'   => array(
				'label'       => __( 'Choose which user(s) to exclude', 'ywdpd' ),
				'type'        => 'custom',
				'action'      => 'ywdpd_show_custom_field',
				'ywdpd_type'  => 'search-customers',
				'desc'        => __( 'Search for the users you want to exclude from this rule', 'ywdpd' ),
				'multiple'    => true,
				'allow_clear' => true,
				'required'    => true,
				'data'        => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_user_rule_exclude',
								'value' => 'specific_customers',
							),
							array(
								'id'    => '_enable_user_rule_exclude',
								'value' => 'yes',
							),
							array(
								'id'      => '_user_rules',
								'value'   => 'customers_list',
								'compare' => '!=',
							),
						)
					),
				),
			),
			'user_rules_excluded_memberships_list' => $membership_plan_excluded,
		);
	}
}

if ( ! function_exists( 'ywdpd_get_schedule_options' ) ) {

	/**
	 * Return the schedule option
	 *
	 * @param string $rule_type The rule type.
	 *
	 * @return array
	 */
	function ywdpd_get_schedule_options( $rule_type ) {
		switch ( $rule_type ) {
			case 'special_offer':
				$label = __( 'Schedule offer', 'ywdpd' );
				break;
			case 'bulk':
				$label = __( 'Schedule discount', 'ywdpd' );
				break;
			default:
				$label = __( 'Schedule rule', 'ywdpd' );
				break;
		}

		return array(
			'schedule_discount_mode' => array(
				'label'      => $label,
				'type'       => 'custom',
				'ywdpd_type' => 'schedule-rules',
				'action'     => 'ywdpd_show_custom_field',
				'desc'       => __( 'Choose whether to schedule a start and end time for this rule', 'ywdpd' ),
			),
		);
	}
}
if ( ! function_exists( 'ywdpd_get_quantity_mode_options' ) ) {
	/**
	 * Return the Quantity mode options
	 *
	 * @param string $rule_type The rule type.
	 *
	 * @return array[]
	 */
	function ywdpd_get_quantity_mode_options( $rule_type ) {
		switch ( $rule_type ) {
			case 'special_offer':
				$label = __( 'Set offer based on', 'ywdpd' );
				break;
			default:
				$label = __( 'Quantity rule will check', 'ywdpd' );
				break;
		}

		return array(
			'quantity_based' => array(
				'label'   => $label,
				'desc'    => __( 'Set which parameter to check to apply the rule', 'ywdpd' ),
				'type'    => 'select',
				'class'   => 'wc-enhanced-select',
				'options' => array(
					'cart_line'                => __( 'Item quantity in cart line', 'ywdpd' ),
					'single_product'           => __( 'Single product quantity in cart - variations NOT counted', 'ywdpd' ),
					'single_variation_product' => __( 'Single product quantity in cart - variations counted', 'ywdpd' ),
					'cumulative'               => __( 'Total number of products in cart', 'ywdpd' ),
				),
				'std'     => 'cart_line',
			),
		);
	}
}
if ( ! function_exists( 'ywdpd_get_extra_options' ) ) {
	/**
	 * Get the extra options
	 *
	 * @return array
	 */
	function ywdpd_get_extra_options() {
		return array(
			'disable_rule'              => array(
				'label'   => __( 'Disable rule when:', 'ywdpd' ),
				'type'    => 'checkbox-array',
				'options' => array(
					'disable_on_sale'           => __( 'The products are on sale', 'ywdpd' ),
					'disable_with_other_coupon' => __( 'A coupon code is applied', 'ywdpd' ),
				),
			),
			'no_apply_with_other_rules' => array(
				'label' => __( 'Disable other rules with lower priority', 'ywdpd' ),
				'desc'  => __( 'Enable to apply this rule and disable other rules with lower priority that are applied to the same products', 'ywdpd' ),
				'type'  => 'onoff',
				'std'   => 'yes',
			),
		);
	}
}

if ( ! function_exists( 'ywdpd_get_show_in_loop_option' ) ) {
	/**
	 * Return the show in loop option
	 *
	 * @return array[]
	 */
	function ywdpd_get_show_in_loop_option() {
		return array(
			'show_in_loop' => array(
				'label' => __( 'Show discount in loop', 'ywdpd' ),
				'desc'  => __( 'Enable if you want to show the discounted price in the loop', 'ywdpd' ),
				'type'  => 'onoff',
				'std'   => 'yes',
			),
		);
	}
}

if ( ! function_exists( 'ywdpd_get_note_to_apply_options' ) ) {

	/**
	 * Return the "Apply note" option
	 *
	 * @param string $rule_type The rule type.
	 *
	 * @return array
	 */
	function ywdpd_get_note_to_apply_options( $rule_type ) {

		switch ( $rule_type ) {
			case 'special_offer':
				$label = __( 'Add a custom message in products with the special offer', 'ywdpd' );
				break;
			case 'bogo':
				$label = __( 'Add a custom message in products with the BOGO offer', 'ywdpd' );
				break;
			case 'bulk':
				$label = __( 'Add a custom message in products with the quantity rule', 'ywdpd' );
				break;
			default:
				$label = __( 'Add a custom message in products with the discount applied', 'ywdpd' );
				break;
		}

		$extra_desc_for_wpml = ywdpd_get_extra_desc_for_wpml();

		return array(
			'show_table_note_apply_to' => array(
				'label' => __( 'Show a custom notice in products with this rule', 'ywdpd' ),
				'desc'  => __( 'Enable to set a custom notice to show in all products in which this rule is applied', 'ywdpd' ),
				'type'  => 'onoff',
				'std'   => 'no',
			),
			'table_note_apply_to'      => array(
				// @since 3.0.0
				'label'   => $label,
				'desc'    => __( 'Enter a custom text to show in the product page where this rule is applied. Use the placeholder <code>%items_with_rule_applied%</code> to show products, categories, or tags to which the discount rule is applied.', 'ywdpd' ) . $extra_desc_for_wpml,
				'type'    => 'textarea-editor',
				'wpautop' => false,
				'data'    => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_show_table_note_apply_to',
								'value' => 'yes',
							),
						)
					),
				),
			),
		);
	}
}

if ( ! function_exists( 'ywdpd_get_note_adjustment_to_options' ) ) {
	/**
	 * Return the "Adjustment note" option
	 *
	 * @return array
	 */
	function ywdpd_get_note_adjustment_to_options() {
		$extra_desc_for_wpml = ywdpd_get_extra_desc_for_wpml();

		return array(
			'show_note_adjustment_to'  => array(
				'label' => __( 'Show a custom notice in products where the discount is applied', 'ywdpd' ),
				'desc'  => __( 'Enable to set a custom notice to show in all products in which the discount is applied', 'ywdpd' ),
				'type'  => 'onoff',
				'std'   => 'no',
				'data'  => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_active_apply_discount_to',
								'value' => 'yes',
							),
						)
					),
				),
			),
			'table_note_adjustment_to' => array(
				'label'   => __( 'Add a custom message in products where the discount is applied', 'ywdpd' ),
				// @since 3.0.0
				'desc'    => __( 'Enter a custom text to show in the product page where the discount is applied. Use the placeholder <code>%required_items%</code> to show products, categories, or tags that users must add to the cart to apply the discount rule to the product for which it has been created', 'ywdpd' ) . $extra_desc_for_wpml,
				'type'    => 'textarea-editor',
				'wpautop' => false,
				'data'    => array(
					'ywdpd-deps' => wp_json_encode(
						array(
							array(
								'id'    => '_active_apply_discount_to',
								'value' => 'yes',
							),
							array(
								'id'    => '_show_note_adjustment_to',
								'value' => 'yes',
							),
						)
					),
				),
			),
		);
	}
}

if ( ! function_exists( 'ywdpd_get_advanced_conditions_options' ) ) {

	/**
	 * Return the advanced conditions configuration
	 *
	 * @param array $modules The modules key to add.
	 *
	 * @return array
	 */
	function ywdpd_get_advanced_conditions_options( $modules ) {
		$default_modules = array(
			'customers'      => __( 'Users', 'ywdpd' ),
			'num_of_orders'  => __( 'Number of orders', 'ywdpd' ),
			'past_expense'   => __( 'Total amount spent', 'ywdpd' ),
			'product'        => __( 'Products', 'ywdpd' ),
			'cart_items'     => __( 'Cart items', 'ywdpd' ),
			'cart_subtotal'  => __( 'Cart subtotal', 'ywdpd' ),
			'country'        => __( 'Customer country', 'ywdpd' ),
			'payment_method' => __( 'Payment method', 'ywdpd' ),
		);

		$modules_enabled = array();
		foreach ( $modules as $module ) {
			if ( isset( $default_modules[ $module ] ) ) {
				$modules_enabled[ $module ] = $default_modules[ $module ];
			}
		}

		$config = array(
			array(
				'id'        => 'condition_name',
				'type'      => 'text',
				'class_row' => 'ywdpd_general_rule',
				'name'      => __( 'Condition name', 'ywdpd' ),
				'desc'      => __( 'Set a name for this cart rule condition', 'ywdpd' ),
			),
			array(
				'id'        => 'condition_for',
				'name'      => __( 'Create a condition based on:', 'ywdpd' ),
				'type'      => 'select',
				'class'     => 'wc-enhanced-select ywdpd_condition_for yith-plugin-fw-select',
				'class_row' => 'ywdpd_general_rule',
				'options'   => $modules_enabled,
				'desc'      => __( 'Choose the condition type', 'ywdpd' ),
			),
		);
		foreach ( $modules as $module ) {
			if ( isset( $default_modules[ $module ] ) ) {
				$function_to_call = 'ywdpd_get_advanced_conditions_' . $module . '_options';
				if ( function_exists( $function_to_call ) ) {
					$config = array_merge( $config, $function_to_call() );
				}
			}
		}

		return $config;
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_customers_options' ) ) {
	/**
	 * Get the customer options
	 *
	 * @return array
	 */
	function ywdpd_get_advanced_conditions_customers_options() {
		/**
		 * APPLY_FILTERS: ywdpd_advanced_conditions_user_include_fields
		 *
		 * Allow change the user options.
		 *
		 * @param array $user_options_include The options
		 *
		 * @return array
		 */
		$user_options_include = apply_filters(
			'ywdpd_advanced_conditions_user_include_fields',
			array(
				array(
					'id'        => 'customers_list',
					'name'      => __( 'Apply discount to these users', 'ywdpd' ),
					'type'      => 'ajax-customers',
					'data'      => array(
						'placeholder'          => esc_attr( __( 'Search for a user', 'ywdpd' ) ),
						'allow_clear'          => false,
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
								array(
									'id'    => 'ywdpd_user_discount_to',
									'value' => 'specific_user_role',
								),
							),
						),
					),
					'multiple'  => true,
					'desc'      => __( 'Select to which users to apply this discount', 'ywdpd' ),
					'class_row' => 'customers specific_user_role',
					'default'   => '',
				),
				array(
					'id'        => 'customers_role_list',
					'name'      => __( 'Apply discount to these user roles', 'ywdpd' ),
					'type'      => 'select',
					'class'     => 'wc-enhanced-select',
					'data'      => array(
						'placeholder'          => esc_attr( __( 'Search for a role', 'ywdpd' ) ),
						'allow_clear'          => false,
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
								array(
									'id'    => 'ywdpd_user_discount_to',
									'value' => 'specific_user_role',
								),
							),
						),
					),
					'multiple'  => true,
					'options'   => ywdpd_get_roles(),
					'desc'      => __( 'Select to which user roles to apply this discount', 'ywdpd' ),
					'class_row' => 'customers specific_user_role',
					'default'   => array(),
				),
			)
		);
		/**
		 * APPLY_FILTERS: ywdpd_advanced_conditions_user_exclude_fields
		 *
		 * Allow change the user options.
		 *
		 * @param array $user_options_exclude The options
		 *
		 * @return array
		 */
		$user_options_exclude = apply_filters(
			'ywdpd_advanced_conditions_user_exclude_fields',
			array(
				array(
					'id'        => 'customers_list_excluded',
					'name'      => __( 'Users excluded', 'ywdpd' ),
					'type'      => 'ajax-customers',
					'data'      => array(
						'placeholder'          => esc_attr( __( 'Search for a user', 'ywdpd' ) ),
						'allow_clear'          => false,
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
								array(
									'id'    => 'ywdpd_user_discount_to',
									'value' => 'all',
								),
								array(
									'id'    => 'ywdpd_enable_exclude_users',
									'value' => 'yes',
								),
							),
						),
					),
					'multiple'  => true,
					'desc'      => __( 'Choose which users to exclude from this discount', 'ywdpd' ),
					'class_row' => 'customers all customers_list_excluded',
					'default'   => '',
				),
				array(
					'id'        => 'customers_role_list_excluded',
					'name'      => __( 'Users roles excluded', 'ywdpd' ),
					'type'      => 'select',
					'class'     => 'wc-enhanced-select',
					'data'      => array(
						'placeholder'          => esc_attr( __( 'Search for a role', 'ywdpd' ) ),
						'allow_clear'          => false,
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
								array(
									'id'    => 'ywdpd_user_discount_to',
									'value' => 'all',
								),
								array(
									'id'    => 'ywdpd_enable_exclude_users',
									'value' => 'yes',
								),
							),
						),
					),
					'multiple'  => true,
					'options'   => ywdpd_get_roles(),
					'desc'      => __( 'Choose which user roles to exclude from this discount', 'ywdpd' ),
					'class_row' => 'customers all customers_list_excluded',
					'default'   => array(),
				),
			)
		);

		return array_merge(
			array(
				array(
					'id'        => 'user_discount_to',
					'name'      => __( 'Apply discount to:', 'ywdpd' ),
					'type'      => 'radio',
					'class'     => 'ywdpd_user_discount_to',
					'options'   => array(
						'all'                => __( 'All users', 'ywdpd' ),
						'specific_user_role' => __( 'Specific users or user roles', 'ywdpd' ),
					),
					'default'   => 'all',
					'desc'      => __( 'Choose to apply the discount to all users (you can exclude users or user roles later) or only to specific users/user roles', 'ywdpd' ),
					'class_row' => 'customers sub_not_hide',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
							)
						),
					),
				),
			),
			$user_options_include,
			array(
				array(
					'id'        => 'enable_exclude_users',
					'name'      => __( 'Exclude users', 'ywdpd' ),
					'type'      => 'onoff',
					'desc'      => __( 'Enable to exclude specific users or user roles from this discount', 'ywdpd' ),
					'default'   => 'no',
					'class_row' => 'customers all subsub_not_hide',
					'class'     => 'ywdpd_enable_exclude_users',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'customers',
								),
								array(
									'id'    => 'ywdpd_user_discount_to',
									'value' => 'all',
								),
							),
						),
					),
				),
			),
			$user_options_exclude
		);
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_num_of_orders_options' ) ) {

	/**
	 * Return the option for amount of orders.
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_num_of_orders_options() {
		return array(
			array(
				'id'         => 'min_max_order',
				'name'       => __( 'Min/Max number of orders', 'ywdpd' ),
				'type'       => 'custom',
				'desc'       => __( 'Set the min/max number of orders required to apply the discount', 'ywdpd' ),
				'ywdpd_type' => 'grid-fields',
				'action'     => 'ywdpd_show_custom_field',
				'headers'    => array(
					'min_order' => __( 'MIN', 'ywdpd' ),
					'max_order' => __( 'MAX', 'ywdpd' ),
				),
				'rows'       => array(
					array(
						'min_order' => array(
							'type'    => 'number',
							'min'     => 1,
							'default' => 1,
							'step'    => 1,
						),
						'max_order' => array(
							'type'    => 'number',
							'min'     => 1,
							'default' => '',
							'step'    => 1,
						),
					),
				),
				'data'       => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'num_of_orders',
							),
						)
					),
				),
				'class_row'  => 'ywdpd-grid-row',
			),
		);

	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_past_expense_options' ) ) {

	/**
	 * Return the option for customer past expense
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_past_expense_options() {
		return array(
			array(
				'id'         => 'min_max_expense',
				/* translators: %s is the woocommerce currency symbol */
				'name'       => sprintf( _x( 'Min/Max past expense (%s)', 'Min/Max past expense ($)', 'ywdpd' ), get_woocommerce_currency_symbol() ),
				'type'       => 'custom',
				'ywdpd_type' => 'grid-fields',
				'action'     => 'ywdpd_show_custom_field',
				'desc'       => __( 'Set the min/max amount a customer must have spent in the past to apply the rule', 'ywdpd' ),
				'headers'    => array(
					'min_order' => __( 'MIN', 'ywdpd' ),
					'max_order' => __( 'MAX', 'ywdpd' ),
				),
				'rows'       => array(
					array(
						'min_expense' => array(
							'type'    => 'number',
							'min'     => 1,
							'default' => 1,
							'step'    => 0.1,
						),
						'max_expense' => array(
							'type'    => 'number',
							'min'     => 1,
							'default' => '',
							'step'    => 0.1,
						),
					),
				),
				'data'       => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'past_expense',
							),
						)
					),
				),
				'class_row'  => 'ywdpd-grid-row',
			),
		);
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_product_options' ) ) {

	/**
	 * Return the option for product condition
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_product_options() {
		/**
		 * APPLY_FILTERS: ywdpd_advanced_conditions_product_include_fields
		 *
		 * Allow change the product options.
		 *
		 * @param array $product_require_options The options
		 *
		 * @return array
		 */
		$product_require_options = apply_filters(
			'ywdpd_advanced_conditions_product_include_fields',
			array(
				array(
					'id'        => 'enable_require_product',
					'name'      => __( 'Require specific products in the cart', 'ywdpd' ),
					'desc'      => __( 'Enable to require specific products in the cart to apply the discount', 'ywdpd' ),
					'default'   => 'yes',
					'type'      => 'onoff',
					'class_row' => 'product require_product',
					'class'     => 'ywdpd_enable_require_product',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'require_product_list',
					'name'      => __( 'Select product', 'ywdpd' ),
					'desc'      => __( 'Choose which products are required in the cart to apply the discount', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-products',
					'data'      => array(
						'placeholder'          => __( 'Search for products', 'ywdpd' ),
						'action'               => 'woocommerce_json_search_products_and_variations',
						'security'             => wp_create_nonce( 'search-products' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product',
									'value' => 'yes',
								),
							)
						),

					),
					'class_row' => 'product require_product enable_require_product_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'require_product_list_mode',
					'name'      => __( 'Apply the discount if:', 'ywdpd' ),
					'desc'      => __( 'Choose to apply the discount when at least one of the specified products is in the cart or only when all of them are in the cart', 'ywdpd' ),
					'type'      => 'radio',
					'options'   => array(
						'at_least'    => __( 'At least one selected product is in the cart', 'ywdpd' ),
						'all_product' => __( 'All selected products are in the cart', 'ywdpd' ),
					),
					'default'   => 'at_least',
					'class_row' => 'product require_product enable_require_product_list',
					'class'     => 'ywdpd_require_product_list_mode',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product',
									'value' => 'yes',
								),
							)
						),
					),
				),
				array(
					'id'        => 'enable_require_product_categories',
					'name'      => __( 'Require specific product categories in the cart', 'ywdpd' ),
					'desc'      => __( 'Enable to require products of specific categories in the cart to apply the discount', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product require_product',
					'class'     => 'ywdpd_enable_require_product_categories',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'require_product_category_list',
					'name'      => __( 'Select product category', 'ywdpd' ),
					'desc'      => __( 'Choose which product categories are required in the cart to apply the discount', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_cat',
						'placeholder'          => __( 'Search for product categories', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product_categories',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product require_product enable_require_product_category_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'require_product_cat_list_mode',
					'name'      => __( 'Apply the discount if:', 'ywdpd' ),
					'desc'      => __( 'Choose to apply the discount when at least one of the specified product categories is in the cart or only when all of them are in the cart', 'ywdpd' ),
					'type'      => 'radio',
					'options'   => array(
						'at_least'     => __( 'At least one of the selected product categories is in the cart', 'ywdpd' ),
						'all_category' => __( 'All selected product categories are in the cart', 'ywdpd' ),
					),
					'default'   => 'at_least',
					'class_row' => 'product require_product enable_require_product_category_list',
					'class'     => 'ywdpd_require_product_cat_list_mode',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product_categories',
									'value' => 'yes',
								),
							)
						),
					),
				),
				array(
					'id'        => 'enable_require_product_tag',
					'name'      => __( 'Require specific product tags in the cart', 'ywdpd' ),
					'desc'      => __( 'Enable to require products of specific tags in the cart to apply the discount', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product require_product',
					'class'     => 'ywdpd_enable_require_product_tag',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'require_product_tag_list',
					'name'      => __( 'Select product tag', 'ywdpd' ),
					'desc'      => __( 'Choose which product tags are required', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_tag',
						'placeholder'          => __( 'Search for product tags', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product_tag',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product require_product enable_require_product_tag_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'require_product_tag_list_mode',
					'name'      => __( 'Apply the discount if:', 'ywdpd' ),
					'desc'      => __( 'Choose to apply the discount when at least one of the specified product tags is in the cart or only when all of them are in the cart', 'ywdpd' ),
					'type'      => 'radio',
					'options'   => array(
						'at_least' => __( 'At least one selected product tag is in cart', 'ywdpd' ),
						'all_tag'  => __( 'All selected product tags are in cart', 'ywdpd' ),
					),
					'default'   => 'at_least',
					'class_row' => 'product require_product enable_require_product_tag_list',
					'class'     => 'ywdpd_require_product_tag_list_mode',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'require_product',
								),
								array(
									'id'    => 'ywdpd_enable_require_product_tag',
									'value' => 'yes',
								),
							)
						),
					),
				),
			)
		);
		/**
		 * APPLY_FILTERS: ywdpd_advanced_conditions_product_exclude_fields
		 *
		 * Allow change the product options.
		 *
		 * @param array $product_require_options The options.
		 *
		 * @return array
		 */
		$product_exclude_options = apply_filters(
			'ywdpd_advanced_conditions_product_exclude_fields',
			array(
				array(
					'id'        => 'enable_exclude_require_product',
					'name'      => __( 'Exclude specific products in cart', 'ywdpd' ),
					'desc'      => __( 'Enable to exclude specific products from discount conditions', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product exclude_product',
					'class'     => 'ywdpd_enable_exclude_require_product',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'exclude_product_list',
					'name'      => __( 'Select product', 'ywdpd' ),
					'desc'      => __( 'Choose which products to exclude from this cart condition validation', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-products',
					'data'      => array(
						'placeholder'          => __( 'Search for products', 'ywdpd' ),
						'action'               => 'woocommerce_json_search_products_and_variations',
						'security'             => wp_create_nonce( 'search-products' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
								array(
									'id'    => 'ywdpd_enable_exclude_require_product',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product exclude_product enable_exclude_product_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'enable_exclude_on_sale_product',
					'name'      => __( 'Exclude \'on-sale products\'', 'ywdpd' ),
					'desc'      => __( 'Enable if you want to exclude \'on-sale products\' from discount conditions', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product exclude_product',
					'class'     => 'ywdpd_enable_exclude_on_sale_product',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'enable_exclude_product_categories',
					'name'      => __( 'Exclude specific product categories from discount validation', 'ywdpd' ),
					'desc'      => __( 'Enable if you want to exclude products of specific categories from discount validation', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product exclude_product',
					'class'     => 'ywdpd_enable_exclude_product_categories',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'exclude_product_category_list',
					'name'      => __( 'Select product category', 'ywdpd' ),
					'desc'      => __( 'Choose which product categories to exclude from condition validation', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_cat',
						'placeholder'          => __( 'Search for product categories', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
								array(
									'id'    => 'ywdpd_enable_exclude_product_categories',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product exclude_product enable_exclude_product_category_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'enable_exclude_product_tag',
					'name'      => __( 'Exclude specific product tags from discount validation', 'ywdpd' ),
					'desc'      => __( 'Enable if you want to exclude products of specific tags from discount validation', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product exclude_product',
					'class'     => 'ywdpd_enable_exclude_product_tag',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'exclude_product_tag_list',
					'name'      => __( 'Select product tag', 'ywdpd' ),
					'desc'      => __( 'Choose which product tags to exclude from condition validation', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_tag',
						'placeholder'          => __( 'Search for product tags', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'exclude_product',
								),
								array(
									'id'    => 'ywdpd_enable_exclude_product_tag',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product exclude_product enable_exclude_product_tag_list',
					'multiple'  => true,
				),
			)
		);
		/**
		 * APPLY_FILTERS: ywdpd_advanced_conditions_product_disable_fields
		 *
		 * Allow change the product options.
		 *
		 * @param array $product_require_options The options
		 *
		 * @return array
		 */
		$product_disable_options = apply_filters(
			'ywdpd_advanced_conditions_product_disable_fields',
			array(
				// Product not selected.
				array(
					'id'        => 'enable_disable_require_product',
					'name'      => __( 'Disable discount when specific products are in cart', 'ywdpd' ),
					'desc'      => __( 'Enable to disable the discount if the user has specific products in the cart', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product disable_product',
					'class'     => 'ywdpd_enable_disable_product',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'disable_product_list',
					'name'      => __( 'Select product', 'ywdpd' ),
					'desc'      => __( 'Choose which products will disable the discount', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-products',
					'data'      => array(
						'placeholder'          => __( 'Search for products', 'ywdpd' ),
						'action'               => 'woocommerce_json_search_products_and_variations',
						'security'             => wp_create_nonce( 'search-products' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
								array(
									'id'    => 'ywdpd_enable_disable_product',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product disable_product enable_disable_product_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'enable_disable_product_categories',
					'name'      => __( 'Disable discount when products of specific categories are in the cart', 'ywdpd' ),
					'desc'      => __( 'Enable to disable the discount if the user has products of specific categories in the cart', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product disable_product',
					'class'     => 'ywdpd_enable_disable_product_categories',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'disable_product_category_list',
					'name'      => __( 'Select product category', 'ywdpd' ),
					'desc'      => __( 'Choose which product categories will disable the discount', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_cat',
						'placeholder'          => __( 'Search for product categories', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
								array(
									'id'    => 'ywdpd_enable_disable_product_categories',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product disable_product enable_disable_product_category_list',
					'multiple'  => true,
				),
				array(
					'id'        => 'disable_exclude_product_tag',
					'name'      => __( 'Disable discount when products with specific tags are in the cart', 'ywdpd' ),
					'desc'      => __( 'Enable to disable the discount if the user has products with specific tags in the cart', 'ywdpd' ),
					'default'   => 'no',
					'type'      => 'onoff',
					'class_row' => 'product disable_product',
					'class'     => 'ywdpd_enable_disable_product_tag',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
							)
						),
					),
				),
				array(
					'id'        => 'disable_product_tag_list',
					'name'      => __( 'Select product tag', 'ywdpd' ),
					'desc'      => __( 'Choose which product tags will disable the discount', 'ywdpd' ),
					'default'   => '',
					'type'      => 'ajax-terms',
					'data'      => array(
						'taxonomy'             => 'product_tag',
						'placeholder'          => __( 'Search for product tags', 'ywdpd' ),
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
								array(
									'id'    => 'ywdpd_product_type',
									'value' => 'disable_product',
								),
								array(
									'id'    => 'ywdpd_enable_disable_product_tag',
									'value' => 'yes',
								),
							)
						),
					),
					'class_row' => 'product disable_product enable_disable_product_tag_list',
					'multiple'  => true,
				),
			)
		);

		return array_merge(
			array(
				array(
					'id'        => 'product_type',
					'name'      => __( 'Condition type', 'ywdpd' ),
					'type'      => 'radio',
					'options'   => array(
						'require_product' => __( 'Require specific products in cart to apply the discount', 'ywdpd' ) . sprintf( '<small>%s</small>', __( 'The discount will be applied only if the user has the specified product(s) in the cart', 'ywdpd' ) ),
						'exclude_product' => __( 'Exclude specific products from the cart condition validation', 'ywdpd' ) . sprintf( '<small>%s</small>', __( 'Excluded products will not be considered for achieving the conditions', 'ywdpd' ) ),
						'disable_product' => __( 'Disable the discount when there is a specific product in the cart', 'ywdpd' ) . sprintf( '<small>%s</small>', __( 'The discount will be not applied if the user has the specified product(s) in the cart', 'ywdpd' ) ),
					),
					'desc'      => __( 'Choose which kind of condition to create based on products', 'ywdpd' ),
					'class_row' => 'product',
					'class'     => 'ywdpd_product_type',
					'default'   => 'require_product',
					'data'      => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'product',
								),
							)
						),
					),
				),
			),
			$product_require_options,
			$product_exclude_options,
			$product_disable_options
		);
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_cart_items_options' ) ) {

	/**
	 * Return the option for amount of cart items condition
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_cart_items_options() {
		$min_max_product_item = array(
			'id'         => 'min_max_product_item',
			'name'       => __( 'Min/Max quantity of product items', 'ywdpd' ),
			'desc'       => __( 'Set the min/max quantity of the same product required in cart to apply the discount', 'ywdpd' ),
			'type'       => 'custom',
			'ywdpd_type' => 'grid-fields',
			'action'     => 'ywdpd_show_custom_field',
			'headers'    => array(
				'min_product_item' => __( 'MIN', 'ywdpd' ),
				'max_product_item' => __( 'MAX', 'ywdpd' ),
			),
			'rows'       => array(
				array(
					'min_product_item' => array(
						'type'    => 'number',
						'min'     => 1,
						'default' => 1,
						'step'    => 1,
					),
					'max_product_item' => array(
						'type'    => 'number',
						'min'     => 1,
						'default' => '',
						'step'    => 1,
					),
				),
			),
			'data'       => array(
				'ywdpd-condition-deps' => wp_json_encode(
					array(
						array(
							'id'    => 'ywdpd_condition_for',
							'value' => 'cart_items',
						),
						array(
							'id'    => 'ywdpd_cart_item_qty_type',
							'value' => 'count_product_items',
						),
					)
				),
			),
			'class_row'  => 'ywdpd-grid-row',
		);
		$min_max_cart_item    = array(
			'id'         => 'min_max_cart_item',
			'name'       => __( 'Min/Max quantity of items in cart', 'ywdpd' ),
			'desc'       => __( 'Set the min/max quantity of items in cart required to apply the discount', 'ywdpd' ),
			'type'       => 'custom',
			'ywdpd_type' => 'grid-fields',
			'action'     => 'ywdpd_show_custom_field',
			'headers'    => array(
				'min_cart_item' => __( 'MIN', 'ywdpd' ),
				'max_cart_item' => __( 'MAX', 'ywdpd' ),
			),
			'rows'       => array(
				array(
					'min_cart_item' => array(
						'type'    => 'number',
						'min'     => 1,
						'default' => 1,
						'step'    => 1,
					),
					'max_cart_item' => array(
						'type'    => 'number',
						'min'     => 1,
						'default' => '',
						'step'    => 1,
					),
				),
			),
			'data'       => array(
				'ywdpd-condition-deps' => wp_json_encode(
					array(
						array(
							'id'    => 'ywdpd_condition_for',
							'value' => 'cart_items',
						),
						array(
							'id'    => 'ywdpd_cart_item_qty_type',
							'value' => 'count_total_cart',
						),
					)
				),
			),
			'class_row'  => 'ywdpd-grid-row',
		);

		return array(
			array(
				'id'        => 'cart_item_qty_type',
				'type'      => 'radio',
				'options'   => array(
					'count_product_items' => __( 'Product items - units of the same product in the cart', 'ywdpd' ),
					'count_total_cart'    => __( 'Cart items - total products in the cart', 'ywdpd' ),
				),
				'class_row' => 'cart_items',
				'class'     => 'ywdpd_cart_item_qty_type',
				'default'   => 'count_product_items',
				'name'      => __( 'Check quantity of', 'ywdpd' ),
				'desc'      => __( 'Choose to link the condition to the number of products or total items in the cart', 'ywdpd' ),
				'data'      => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'cart_items',
							),
						)
					),
				),
			),
			$min_max_product_item,
			$min_max_cart_item,
		);
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_cart_subtotal_options' ) ) {

	/**
	 * Return the option for cart subtotal condition
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_cart_subtotal_options() {
		return array(
			apply_filters(
				'ywdpd_get_advanced_conditions_cart_subtotal_options',	 
				array(
					'id'         => 'min_max_subtotal',
					/* translators: %s is the woocommerce currency symbol */
					'name'       => sprintf( _x( 'Min/Max cart subtotal (%s)', 'Min/Max cart subtotal (%s)', 'ywdpd' ), get_woocommerce_currency_symbol() ),
					'type'       => 'custom',
					'ywdpd_type' => 'grid-fields',
					'action'     => 'ywdpd_show_custom_field',
					'desc'       => __( 'Set the min/max cart subtotal required to apply the discount', 'ywdpd' ),
					'headers'    => array(
						'min_subtotal' => __( 'MIN', 'ywdpd' ),
						'max_subtotal' => __( 'MAX', 'ywdpd' ),
					),
					'rows'       => array(
						array(
							'min_subtotal' => array(
								'type'    => 'number',
								'min'     => 1,
								'default' => 1,
								'step'    => 0.1,
							),
							'max_subtotal' => array(
								'type'    => 'number',
								'min'     => 1,
								'default' => '',
								'step'    => 0.1,
							),
						),
					),
					'data'       => array(
						'ywdpd-condition-deps' => wp_json_encode(
							array(
								array(
									'id'    => 'ywdpd_condition_for',
									'value' => 'cart_subtotal',
								),
							)
						),
					),
					'class_row'  => 'ywdpd-grid-row',
				),
			)
		);
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_country_options' ) ) {

	/**
	 * Return the option for country condition
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_country_options() {
		$country_include_options = array(
			array(
				'id'        => 'enable_include_require_country',
				'name'      => __( 'Include specific country', 'ywdpd' ),
				'desc'      => __( 'Enable to show the deal to specific countries', 'ywdpd' ),
				'default'   => 'no',
				'type'      => 'onoff',
				'class_row' => 'country include_country',
				'class'     => 'ywdpd_enable_include_require_country',
				'data'      => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'country',
							),
						),
					),
				),
			),
			array(
				'id'          => 'include_require_country_list',
				'name'        => __( 'Customer country', 'ywdpd' ),
				'desc'        => __( 'Choose which country to include in this deal condition validation', 'ywdpd' ),
				'default'     => '',
				'type'        => 'select',
				'class'       => 'wc-enhanced-select yith-plugin-fw-select',
				'options'     => WC()->countries->get_countries(),
				'class_row'   => 'country include_country enable_include_require_country_list',
				'multiple'    => true,
				'placeholder' => __( 'Search for a country', 'ywdpd' ),
				'data'        => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'country',
							),
							array(
								'id'    => 'ywdpd_enable_include_require_country',
								'value' => 'yes',
							),
						),
					),
				),
			),

		);
		$country_exclude_options = array(
			array(
				'id'        => 'enable_exclude_require_country',
				'name'      => __( 'Exclude specific country', 'ywdpd' ),
				'desc'      => __( 'Enable to exclude specific countries from this deal', 'ywdpd' ),
				'default'   => 'no',
				'type'      => 'onoff',
				'class_row' => 'country exclude_country',
				'class'     => 'ywdpd_enable_exclude_require_country',
				'data'      => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'country',
							),
						),
					),
				),
			),
			array(
				'id'          => 'exclude_require_country_list',
				'name'        => __( 'Customer country', 'ywdpd' ),
				'desc'        => __( 'Choose which country to exclude from this deal condition validation', 'ywdpd' ),
				'default'     => '',
				'type'        => 'select',
				'class'       => 'wc-enhanced-select yith-plugin-fw-select',
				'options'     => WC()->countries->get_countries(),
				'class_row'   => 'country exclude_country enable_exclude_require_country_list',
				'multiple'    => true,
				'placeholder' => __( 'Search for a country', 'ywdpd' ),
				'data'        => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'country',
							),
							array(
								'id'    => 'ywdpd_enable_exclude_require_country',
								'value' => 'yes',
							),
						),
					),
				),
			),
		);

		return array_merge( $country_include_options, $country_exclude_options );
	}
}
if ( ! function_exists( 'ywdpd_get_advanced_conditions_payment_method_options' ) ) {
	/**
	 * Return the option for the payment method condition
	 *
	 * @return array[]
	 */
	function ywdpd_get_advanced_conditions_payment_method_options() {
		$all_gateways = WC()->payment_gateways()->get_available_payment_gateways();
		$options      = array();
		foreach ( $all_gateways as $gateway_id => $gateway ) {
			$options[ $gateway_id ] = $gateway->get_method_title();
		}
		$payment_method_include_options = array(
			array(
				'id'      => 'enable_include_require_payment_method',
				'name'    => __( 'Include specific payment method', 'ywdpd' ),
				'desc'    => __( 'Enable to apply the discount if a specific payment method is selected', 'ywdpd' ),
				'default' => 'no',
				'type'    => 'onoff',
				'class'   => 'ywdpd_enable_include_require_payment_method',
				'data'    => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'payment_method',
							),
						),
					),
				),
			),
			array(
				'id'          => 'include_require_payment_method_list',
				'name'        => __( 'Payment method', 'ywdpd' ),
				'desc'        => __( 'Choose which payment method to include in this condition validation', 'ywdpd' ),
				'default'     => '',
				'type'        => 'select',
				'class'       => 'wc-enhanced-select yith-plugin-fw-select',
				'options'     => $options,
				'multiple'    => true,
				'placeholder' => __( 'Search for a payment method', 'ywdpd' ),
				'data'        => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'payment_method',
							),
							array(
								'id'    => 'ywdpd_enable_include_require_payment_method',
								'value' => 'yes',
							),
						),
					),
				),
			),

		);
		$payment_method_exclude_options = array(
			array(
				'id'      => 'enable_exclude_require_payment_method',
				'name'    => __( 'Exclude specific payment method', 'ywdpd' ),
				'desc'    => __( 'Enable to not apply the discount if a specific payment method is selected', 'ywdpd' ),
				'default' => 'no',
				'type'    => 'onoff',
				'class'   => 'ywdpd_enable_exclude_require_payment_method',
				'data'    => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'payment_method',
							),
						),
					),
				),
			),
			array(
				'id'          => 'exclude_require_payment_method_list',
				'name'        => __( 'Payment method', 'ywdpd' ),
				'desc'        => __( 'Choose which payment method to exclude from this condition validation', 'ywdpd' ),
				'default'     => '',
				'type'        => 'select',
				'class'       => 'wc-enhanced-select yith-plugin-fw-select',
				'options'     => $options,
				'multiple'    => true,
				'placeholder' => __( 'Search for a payment method', 'ywdpd' ),
				'data'        => array(
					'ywdpd-condition-deps' => wp_json_encode(
						array(
							array(
								'id'    => 'ywdpd_condition_for',
								'value' => 'payment_method',
							),
							array(
								'id'    => 'ywdpd_enable_exclude_require_payment_method',
								'value' => 'yes',
							),
						),
					),
				),
			),

		);

		return array_merge( $payment_method_include_options, $payment_method_exclude_options );
	}
}