<?php
/**
 * General functions.
 *
 * @package YITH WooCommerce Dynamic Pricing & Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH <plugins@yithemes.com>
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}


if ( ! function_exists( 'ywdpd_have_dynamic_coupon' ) ) {
	/**
	 * Check if cart has dynamic coupon.
	 *
	 * @return bool
	 */
	function ywdpd_have_dynamic_coupon() {

		$coupons = WC()->cart->get_coupons();

		if ( empty( $coupons ) ) {
			return false;
		}

		$dynamic_coupon = YITH_WC_Dynamic_Options::get_coupon_label();

		foreach ( $coupons as $code => $value ) {
			if ( strtolower( $code ) === strtolower( $dynamic_coupon ) ) {
				return true;
			}
		}

		return false;
	}
}

if ( ! function_exists( 'ywdpd_get_discounted_price_table' ) ) {
	/**
	 * Get the discounted price table.
	 *
	 * @param float $price Price.
	 * @param array $rule Rule.
	 *
	 * @return int
	 */
	function ywdpd_get_discounted_price_table( $price, $rule ) {

		if ( empty( $price ) || ! isset( $rule['type_discount'] ) ) {
			return $price;
		}

		$discount        = 0;
		$discount_amount = str_replace( ',', '.', $rule['discount_amount'] );
		if ( 'percentage' === $rule['type_discount'] ) {
			$discount = $price * $discount_amount;
		} elseif ( 'price' === $rule['type_discount'] ) {
			$discount = $discount_amount;
		} elseif ( 'fixed-price' === $rule['type_discount'] ) {
			/**
			 * APPLY_FILTERS: ywdpd_maybe_should_be_converted
			 *
			 * Is possible change the discount amount. Useful for multicurrency plugins.
			 *
			 * @param float $discount_amount the discount amount.
			 *
			 * @return float
			 */
			$discount = $price - apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
		}

		$new_price = ( ( $price - $discount ) < 0 ) ? 0 : ( $price - $discount );

		/**
		 * APPLY_FILTERS: ywdp_discounted_price_table
		 *
		 * Change the discounted price in the table.
		 *
		 * @param float $new_price the new price.
		 * @param float $price the old price.
		 * @param array $rule The rule configuration.
		 *
		 * @return float
		 */
		return apply_filters( 'ywdp_discounted_price_table', $new_price, $price, $rule );
	}
}

if ( ! function_exists( 'ywdpd_check_cart_coupon' ) ) {
	/**
	 * Check if cart have already coupon applied
	 *
	 * @since  1.1.4
	 */
	function ywdpd_check_cart_coupon() {

		if ( ! WC()->cart ) {
			return false;
		}

		$cart_coupons = WC()->cart->applied_coupons;

		if ( ywdpd_have_dynamic_coupon() ) {
			return false;
		}

		/**
		 * APPLY_FILTERS: ywdpd_check_cart_coupon
		 *
		 * Cart has coupon.
		 *
		 * @param bool $has_coupon True or false.
		 *
		 * @return bool
		 */
		return apply_filters( 'ywdpd_check_cart_coupon', ! empty( $cart_coupons ) );
	}
}

if ( ! function_exists( 'yit_wpml_object_id' ) ) {
	/**
	 * Get id of post translation in current language
	 *
	 * @param int         $element_id Element id.
	 * @param string      $element_type Element type.
	 * @param bool        $return_original_if_missing Bool.
	 * @param null|string $ulanguage_code Language code.
	 *
	 * @return int the translation id
	 * @since  2.0.0
	 */
	function yit_wpml_object_id( $element_id, $element_type = 'post', $return_original_if_missing = false, $ulanguage_code = null ) {
		if ( function_exists( 'wpml_object_id_filter' ) ) {
			return wpml_object_id_filter( $element_id, $element_type, $return_original_if_missing, $ulanguage_code );
		} elseif ( function_exists( 'icl_object_id' ) ) {
			return icl_object_id( $element_id, $element_type, $return_original_if_missing, $ulanguage_code );
		} else {
			return $element_id;
		}
	}
}

if ( ! function_exists( 'ywdpd_get_note' ) ) {
	/**
	 * Get the note.
	 *
	 * @param string $note Note.
	 *
	 * @return mixed|void
	 */
	function ywdpd_get_note( $note ) {
		if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
			global $sitepress;
			$current_language = $sitepress->get_current_language();
			$epression_rule   = '/(?<=\[' . $current_language . '\])(\s*.*\s*)(?=\[\/' . $current_language . '\])/';
			if ( preg_match( $epression_rule, $note, $match ) ) {
				$note = $match[0];
			}
		}

		/**
		 * APPLY_FILTERS: ywdpd_get_note
		 *
		 * Change the note.
		 *
		 * @param string $note The note.
		 *
		 * @return string
		 */
		return apply_filters( 'ywdpd_get_note', $note );
	}
}

if ( ! function_exists( 'ywdpd_coupon_is_valid' ) ) {

	/**
	 * Check if a coupon is valid
	 *
	 * @param mixed $coupon Coupon.
	 * @param array $object Object.
	 *
	 * @return bool|WP_Error
	 * @throws Exception Return the error.
	 */
	function ywdpd_coupon_is_valid( $coupon, $object = array() ) {
		if ( version_compare( WC()->version, '3.2.0', '>=' ) ) {
			$wc_discounts = new WC_Discounts( $object );
			$valid        = $wc_discounts->is_coupon_valid( $coupon );
			$valid        = is_wp_error( $valid ) ? false : $valid;
		} else {
			$valid = $coupon->is_valid();
		}
		/**
		 * APPLY_FILTERS: yith_ywdpd_coupon_is_valid
		 *
		 * The coupon is valid.
		 *
		 * @param bool      $valid True or false.
		 * @param WC_Coupon $coupon The coupon.
		 * @param array     $object The coupon data.
		 *
		 * @return string
		 */
		$valid = apply_filters( 'yith_ywdpd_coupon_is_valid', $valid, $coupon, $object );

		return $valid;
	}
}


if ( ! function_exists( 'ywdpd_check_valid_admin_page' ) ) {
	/**
	 * Return if the current pagenow is valid for a post_type, useful if you want add metabox, scripts inside the editor of a particular post type
	 *
	 * @param string $post_type_name Post Type Name.
	 *
	 * @return bool
	 */
	function ywdpd_check_valid_admin_page( $post_type_name ) {
		global $pagenow;

		$posted = $_REQUEST; // phpcs:ignore WordPress.Security.NonceVerification
		$post   = isset( $posted['post'] ) ? $posted['post'] : ( isset( $posted['post_ID'] ) ? $posted['post_ID'] : 0 );
		$post   = get_post( $post );

		if ( ( $post && $post->post_type === $post_type_name ) || ( isset( $posted['post_type'] ) && $post_type_name === $posted['post_type'] ) ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'yith_ywdpd_check_update_to_cpt' ) ) {
	/**
	 * Check if is necessary transform the rules from option to cpt
	 *
	 */
	function yith_ywdpd_check_update_to_cpt() {

		$ywdpd_updated_to_cpt = get_option( 'ywdpd_updated_to_cpt' );
		$check                = true === $ywdpd_updated_to_cpt || 1 === $ywdpd_updated_to_cpt || '1' === $ywdpd_updated_to_cpt || 'yes' === $ywdpd_updated_to_cpt;
		if ( ! $check && get_option( 'yit_ywdpd_options' ) ) {
			yith_ywdpd_update_to_cpt();
		}
	}
}

if ( ! function_exists( 'yith_ywdpd_update_to_cpt' ) ) {

	/**
	 * Transforms the old rules in Custom post type
	 *
	 */
	function yith_ywdpd_update_to_cpt() {
		$option_types = array( 'pricing', 'cart' );
		$options      = get_option( 'yit_ywdpd_options' );
		$args         = array(
			'post_type'      => 'ywdpd_discount',
			'comment_status' => 'closed',
			'post_status'    => 'publish',
		);
		if ( $options ) {

			foreach ( $option_types as $type ) {
				$priority           = 0;
				$cart_discount_rule = array();
				if ( isset( $options[ $type . '-rules' ] ) ) {
					$rules = $options[ $type . '-rules' ];

					foreach ( $rules as $key => $value ) {
						$priority ++;
						$args['post_title'] = $value['description'];

						$id = wp_insert_post( $args );
						if ( $id ) {
							add_post_meta( $id, '_key', $key );
							add_post_meta( $id, '_discount_type', $type );
							add_post_meta( $id, '_priority', $priority );
							foreach ( $value as $_key => $item ) {
								if ( 'cart' === $type ) {
									if ( 'discount_type' === $_key || 'discount_amount' === $_key ) {
										$cart_discount_rule[ $_key ] = $item;
									}
								}
								$meta_key = in_array( $_key, array( 'rules', 'so-rule' ), true ) ? $_key : '_' . $_key;
								add_post_meta( $id, $meta_key, $item );
							}

							! empty( $cart_discount_rule ) && add_post_meta( $id, '_discount_rule', $cart_discount_rule );
						}
					}
				}
			}

			update_option( 'ywdpd_updated_to_cpt', 'yes' );
		}
	}
}

if ( ! function_exists( 'ywdpd_recover_rules' ) ) {
	/**
	 * Recover the rules from cache.
	 *
	 * @param string $type Discount type.
	 *
	 * @return array
	 * @deprecated 3.0.0
	 */
	function ywdpd_recover_rules( $type ) {
		yith_ywdpd_deprecated_function( 'ywdpd_recover_rules', '3.0', 'use ywdpd_get_cart_rules or ywdpd_get_price_rules instead' );

		if ( 'pricing' === $type ) {
			$rules = ywdpd_get_price_rules();
		} else {
			$rules = ywdpd_get_cart_rules();
		}

		return $rules;
	}
}

if ( ! function_exists( 'ywdpd_get_last_priority' ) ) {
	/**
	 * Returns the last priority
	 *
	 * @return int|mixed
	 */
	function ywdpd_get_last_priority() {

		$args = array(
			'post_type'      => 'ywdpd_discount',
			'post_status'    => 'publish',
			'posts_per_page' => 1,
			'orderby'        => 'meta_value_num',
			'meta_key'       => '_priority', // phpcs:ignore WordPress.DB.SlowDBQuery
			'order'          => 'DESC',
		);

		$posts = new WP_Query( $args );

		return ( $posts->post ) ? get_post_meta( $posts->post->ID, '_priority', true ) : 0;

	}
}

if ( ! function_exists( 'ywdpd_get_roles' ) ) {
	/**
	 * Get user roles
	 *
	 * @return array
	 */
	function ywdpd_get_roles() {
		global $wp_roles;
		$roles = array();

		foreach ( $wp_roles->get_names() as $key => $role ) {
			$roles[ $key ] = translate_user_role( $role );
		}

		return array_merge(
			array(
				'guest' => __(
					'Guest',
					'ywdpd'
				),
			),
			$roles
		);
	}
}

if ( ! function_exists( 'yith_dynamic_pricing_add_item_popup' ) ) {
	/**
	 * Show the add item popup
	 *
	 * @param array  $items The items.
	 * @param string $rule_key The rule key.
	 * @param string $type The type.
	 * @param array  $item_to_show The item to show.
	 */
	function yith_dynamic_pricing_add_item_popup( $items, $rule_key, $type, $item_to_show ) {

		$args = array(
			'item_ids'   => $items['item_ids'],
			'list_class' => $type,
			'rule_key'   => $rule_key,
		);
		if ( 'product_ids' === $items['type'] ) {
			$args['item_class'] = 'product';
			$args['max_units']  = $item_to_show['max_units'] ?? - 1;
			$args['countdown']  = $item_to_show['countdown'] ?? array();
		} else {
			$args['item_class']    = 'product_taxonomy';
			$args['taxonomy_name'] = $items['type'];
		}

		if ( 'special_offer' === $type ) {
			$args['total_to_add'] = $item_to_show['total_to_add'];
		}

		wc_get_template( 'yith_ywdpd_popup_' . $args['item_class'] . '_items.php', $args, YITH_YWDPD_TEMPLATE_PATH, YITH_YWDPD_TEMPLATE_PATH );

	}

	add_action( 'ywdpd_show_popup_items', 'yith_dynamic_pricing_add_item_popup', 10, 4 );
}

if ( ! function_exists( 'yith_dynamic_is_variation_attributes_set' ) ) {

	/**
	 * Check if a variation has all attributes set.
	 *
	 * @param WC_Product_Variation $variation The product object.
	 *
	 * @return bool
	 * @since  2.1.3
	 */
	function yith_dynamic_is_variation_attributes_set( $variation ) {
		$has_all_attributes = true;
		$attributes         = $variation->get_attributes();

		foreach ( $attributes as $key => $value ) {

			if ( empty( $value ) ) {
				$has_all_attributes = false;
				break;
			}
		}

		return $has_all_attributes;
	}
}


if ( ! function_exists( 'ywdpd_get_rules' ) ) {
	/**
	 * Get all rules
	 *
	 * @param false|string|string[] $types The dynamic rule to return.
	 *
	 * @return array
	 * @since  3.0.0
	 */
	function ywdpd_get_rules( $types = false ) {
		$args = array(
			'post_type'      => 'ywdpd_discount',
			'post_status'    => 'publish',
			'posts_per_page' => - 1,
		);

		if ( $types ) {
			$types              = ! is_array( $types ) ? array( $types ) : $types;
			$args['meta_query'] = array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				array(
					'key'     => '_discount_mode',
					'value'   => $types,
					'compare' => 'IN',
				),
			); // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		}

		$query = new WP_Query( $args );
		$rules = array_filter( array_map( 'ywdpd_get_rule', $query->posts ) );

		return $rules;
	}
}

if ( ! function_exists( 'ywdpd_get_rule' ) ) {
	/**
	 * Get the rule
	 *
	 * @param WP_Post|int $the_rule The post object or the post id.
	 *
	 * @return YWDPD_Rule|bool Rule object or false if the product cannot be loaded.
	 * @since  3.0.0
	 */
	function ywdpd_get_rule( $the_rule ) {
		$rule_id = false;

		if ( is_numeric( $the_rule ) ) {
			$rule_id = $the_rule;
		} elseif ( $the_rule instanceof YWDPD_Rule ) {
			$rule_id = $the_rule->get_id();
		} elseif ( ! empty( $the_rule->ID ) ) {
			$rule_id = absint( $the_rule->ID );
		}

		if ( ! $rule_id ) {
			return false;
		}


		$subtype = get_post_meta( $rule_id, '_discount_mode', true );

		if ( 'bulk' === $subtype ) {
			$classname = 'YWDPD_Quantity_Table';
		} else {
			if ( 'cart' === $subtype ) {
				$classname = 'YWDPD_Cart_Rule';
			} else {
				$classname = 'YWDPD_' . implode( '_', array_map( 'ucfirst', explode( '_', $subtype ) ) );
			}
		}

		if ( ! class_exists( $classname ) ) {
			$classname = 'YWDPD_Quantity_Table';
		}
		try {
			return new $classname( $rule_id );
		} catch ( Exception $e ) {
			return false;
		}
	}
}

if ( ! function_exists( 'ywdpd_filter_active_rules' ) ) {
	/**
	 * Return only the active rules.
	 *
	 * @param YWDPD_Rule $rule The price rule.
	 *
	 * @return bool
	 * @since  3.0.0
	 */
	function ywdpd_filter_active_rules( $rule ) {
		return $rule->is_enabled();
	}
}

if ( ! function_exists( 'ywdpd_sort_rule_by_priority' ) ) {
	/**
	 * Sort the price rule by priority DESC
	 *
	 * @param YWDPD_Rule $a The first object to compare.
	 * @param YWDPD_Rule $b The second object to compare.
	 *
	 * @return int
	 * @since  3.0.0
	 */
	function ywdpd_sort_rule_by_priority( $a, $b ) {

		return $a->get_priority() <=> $b->get_priority();
	}
}

if ( ! function_exists( 'ywdpd_sort_cart_by_price' ) ) {
	/**
	 * Sort the cart by price asc ( from cheapest to most expensive )
	 *
	 * @param array $cart_item_a The first object to compare.
	 * @param array $cart_item_b The second object to compare.
	 *
	 * @return int
	 * @since  3.0.0
	 */
	function ywdpd_sort_cart_by_price( $cart_item_a, $cart_item_b ) {

		if ( empty( $cart_item_a ) || empty( $cart_item_b ) ) {
			return 0;
		} else {
			$product_a = $cart_item_a['data'] ?? false;
			$product_b = $cart_item_b['data'] ?? false;

			if ( $product_a && $product_b ) {
				return $product_a->get_price( 'edit' ) <=> $product_b->get_price( 'edit' );
			} else {
				return 0;
			}
		}
	}
}
if ( ! function_exists( 'ywdpd_clone_cart' ) ) {
	/**
	 * Clone the actual cart.
	 *
	 * @param WC_Cart|null $cart The cart object.
	 *
	 * @return array
	 * @since  3.0.0
	 */
	function ywdpd_clone_cart( $cart = null ) {
		$cloned_cart = array();

		if ( is_null( $cart ) && ! is_null( WC()->cart ) ) {
			$cart = WC()->cart;
		}

		/**
		 * APPLY_FILTERS: ywdpd_skip_cart_check
		 *
		 * Skip cart check.
		 *
		 * @param bool    $skip True or false.
		 * @param WC_Cart $cart The cart.
		 *
		 * @return bool
		 */
		if ( $cart instanceof WC_Cart && ! $cart->is_empty() && ! apply_filters( 'ywdpd_skip_cart_check', false, $cart ) ) {
			foreach ( $cart->get_cart_contents() as $cart_item_key => $cart_item ) {
				/**
				 * APPLY_FILTERS: ywdpd_add_cart_item_in_clone
				 *
				 * Process cart item check.
				 *
				 * @param bool  $process True or false.
				 * @param array $cart_item The cart item.
				 *
				 * @return bool
				 */
				if ( apply_filters( 'ywdpd_add_cart_item_in_clone', true, $cart_item ) ) {
					$cloned_cart[ $cart_item_key ] = $cart_item;
				}
			}
		}

		return $cloned_cart;
	}
}

if ( ! function_exists( 'ywdpd_get_price_rules' ) ) {
	/**
	 * Return all active price rules, sorted by priority
	 *
	 * @return YWDPD_Price_Rule[]
	 * @since  3.0.0
	 */
	function ywdpd_get_price_rules( $types = array() ) {
		$price_rules   = get_transient( 'ywdpd_get_price_rules' );
		$default_types = count( $types ) > 0 ? $types : array(
			'bulk',
			'special_offer',
			'discount_whole',
			'category_discount',
			'bogo',
			'last_deals',
			'gift_products',
		);

		if ( false === $price_rules ) {

			$price_rules = ywdpd_get_dynamic_rules_by_priority( $default_types );

			set_transient( 'ywdpd_get_price_rules', $price_rules );
		}

		return $price_rules;
	}
}
if ( ! function_exists( 'ywdpd_get_dynamic_rules_by_priority' ) ) {
	/**
	 * Return the rules order by priority
	 *
	 * @param bool|string[]|string $types The rule type ( pricing or cart ).
	 * @param bool                 $only_active Return only active rule.
	 *
	 * @return array
	 * @since  3.0.0
	 */
	function ywdpd_get_dynamic_rules_by_priority( $types, $only_active = true ) {
		$price_rules = ywdpd_get_rules( $types );

		if ( $only_active ) {
			$price_rules = array_filter( $price_rules, 'ywdpd_filter_active_rules' );
		}
		uasort( $price_rules, 'ywdpd_sort_rule_by_priority' );

		return $price_rules;
	}
}
if ( ! function_exists( 'ywdpd_get_cart_rules' ) ) {
	/**
	 * Return all active cart rules
	 *
	 * @return YWDPD_Cart_Rule[]
	 * @since  3.0.0
	 */
	function ywdpd_get_cart_rules() {
		$cart_rules = get_transient( 'ywpdp_get_cart_rules' );
		if ( false === $cart_rules ) {
			$cart_rules = ywdpd_get_dynamic_rules_by_priority( 'cart' );
			set_transient( 'ywpdp_get_cart_rules', $cart_rules );
		}

		return $cart_rules;
	}
}

if ( ! function_exists( 'ywdpd_get_price_rules_by_type' ) ) {
	/**
	 * Return the price rules by types
	 *
	 * @param string|array $types The rule type to return.
	 *
	 * @return YWDPD_Price_Rule[]
	 * @since  3.0.0
	 */
	function ywdpd_get_price_rules_by_type( $types = false ) {
		return ywdpd_get_dynamic_rules_by_priority( $types );
	}
}


if ( ! function_exists( 'ywdpd_is_quantity_table_template_customized' ) ) {
	/**
	 * Check if there is the template customized in the theme.
	 *
	 * @return bool
	 * @since  3.0.0
	 */
	function ywdpd_is_quantity_table_template_customized() {
		return file_exists( get_stylesheet_directory() . '/yith_ywdpd_table_pricing.php' );
	}
}


if ( ! function_exists( 'ywdpd_delete_plugin_transient' ) ) {
	/**
	 * Delete the plugin transient
	 *
	 * @since  3.0.0
	 */
	function ywdpd_delete_plugin_transient() {
		delete_transient( 'ywdpd_get_price_rules' );
		delete_transient( 'ywpdp_get_cart_rules' );
	}
}

if ( ! function_exists( 'ywdpd_get_price_rule_type' ) ) {
	/**
	 * Return all rule type
	 *
	 * @param bool|string $rule_type If is a string return the rule type label.
	 *
	 * @return array|string
	 */
	function ywdpd_get_price_rule_type( $rule_type = false ) {
		$types = array(
			'bulk'              => __( 'Qty discount', 'ywdpd' ),
			'special_offer'     => __( 'Special offer (3x2, 2x1, etc.)', 'ywdpd' ),
			'bogo'              => __( 'Buy 1 Get 1', 'ywdpd' ), // @since 3.0.0
			'gift_products'     => __( 'Free gift on cart', 'ywdpd' ),
			'discount_whole'    => __( 'Global discount', 'ywdpd' ),
			'category_discount' => __( 'Category discount', 'ywdpd' ),
			'last_deals'        => __( 'Last deal on checkout page', 'ywdpd' ),
			'cart'              => __( 'Cart discount', 'ywdpd' ),
		);
		if ( $rule_type ) {
			return $types[ $rule_type ] ?? false;
		} else {
			return $types;
		}

	}
}

if ( ! function_exists( 'ywdpd_set_modal_rule_as_visited' ) ) {

	/**
	 * Set a modal rule as visited in the cookie
	 *
	 * @param int $rule_id The rule id.
	 *
	 * @return void
	 */
	function ywdpd_set_modal_rule_as_visited( $rule_id ) {
		$value = array();
		if ( ! empty( $_COOKIE['yith_ywdpd_modal_rule'] ) ) {
			$value = json_decode( wp_unslash( $_COOKIE['yith_ywdpd_modal_rule'] ), true ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		}

		$value[] = $rule_id;
		$value   = wp_json_encode( stripslashes_deep( $value ) );
		$time    = time() + DAY_IN_SECONDS;
		wc_setcookie( 'yith_ywdpd_modal_rule', $value, $time, false );
	}
}

if ( ! function_exists( 'ywdpd_is_modal_rule_visited' ) ) {

	/**
	 * Check if a modal rule has been visited or not
	 *
	 * @param int $rule_id The rule id.
	 *
	 * @return bool
	 */
	function ywdpd_is_modal_rule_visited( $rule_id ) {
		$is_visited = false;
		if ( ! empty( $_COOKIE['yith_ywdpd_modal_rule'] ) ) {
			$value = json_decode( wp_unslash( $_COOKIE['yith_ywdpd_modal_rule'] ), true ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

			$is_visited = in_array( $rule_id, $value ); // phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
		}

		return apply_filters( 'ywdpd_modal_rule_is_visited', $is_visited, $rule_id );
	}
}

if ( ! function_exists( 'ywdpd_clear_modal_rule_cookie' ) ) {
	/**
	 * Remove cookie from cache
	 *
	 * @since  4.0.0
	 */
	function ywdpd_clear_modal_rule_cookie() {
		$value                            = wp_json_encode( stripslashes_deep( array() ) );
		$_COOKIE['yith_ywdpd_modal_rule'] = $value;
		wc_setcookie( 'yith_ywdpd_modal_rule', $value, time() - 3600, false );
	}
}

if ( ! function_exists( 'ywdpd_sync_dynamic_rules_batch' ) ) {
	/**
	 * Check if we're sync dynamic rules
	 *
	 * @return bool
	 */
	function ywdpd_sync_dynamic_rules_batch() {
		$table_updates_pending = WC()->queue()->search(
			array(
				'status'   => 'pending',
				'group'    => 'ywdpd_process_rule_group',
				'per_page' => 1,
			)
		);
		$table_fix_pending     = WC()->queue()->search(
			array(
				'status'   => 'pending',
				'group'    => 'ywdpd-db-update-process-fix-rules',
				'per_page' => 1,
			)
		);

		return (bool) ( count( $table_updates_pending ) || count( $table_fix_pending ) );
	}
}


if ( ! function_exists( 'yith_ywdpd_print_notice' ) ) {
	/**
	 * Print notice
	 *
	 * @param string $notice The notice.
	 * @param string $type Type.
	 * @param false  $dismissible Dismissible flag.
	 * @param string $key The key.
	 */
	function yith_ywdpd_print_notice( $notice, $type = 'info', $dismissible = false, $key = '' ) {
		if ( ! $key ) {
			$key = md5( $notice . '_' . $type );
		}
		$key    = sanitize_key( $key );
		$cookie = 'yith_ywdpd_notice_dismiss_' . $key;
		$id     = 'yith-ywpdp-notice-' . $key;

		if ( $dismissible && ! empty( $_COOKIE[ $cookie ] ) ) {
			return;
		}

		yith_plugin_fw_get_component(
			array(
				'id'          => $id,
				'type'        => 'notice',
				'notice_type' => $type,
				'message'     => $notice,
				'inline'      => false,
				'dismissible' => $dismissible,
			),
			true
		);

		if ( $dismissible ) {
			?>
            <script>
                jQuery('#<?php echo esc_attr( $id ); ?>').on('click', '.yith-plugin-fw__notice__dismiss', function () {
                    var expires = (new Date(Date.now() + (15 * 60 * 1000))).toUTCString();
                    document.cookie = "<?php echo esc_attr( $cookie ); ?>=1; expires=" + expires + ";";
                });
            </script>
			<?php
		}
	}
}


if ( ! function_exists( 'yith_ywdpd_get_total_discount_message' ) ) {
	function yith_ywdpd_get_total_discount_message() {
		$message          = '';
		$original_message = YITH_WC_Dynamic_Options::get_discount_info_message();

		$coupons_applied = WC()->cart->get_applied_coupons();
		$tax_excluded    = 'tax_excluded' === YITH_WC_Dynamic_Options::how_calculate_discounts();
		$amount          = 0;

		foreach ( $coupons_applied as $coupon_code ) {

			$coupon = new WC_Coupon( $coupon_code );
			$meta   = $coupon->get_meta( 'ywdpd_coupon' );
			if ( ! empty( $meta ) ) {
				$amount = WC()->cart->get_coupon_discount_amount( $coupon->get_code(), $tax_excluded );
				break;
			}
		}

		$subtotal = WC()->cart->get_subtotal();
		if ( ! $tax_excluded ) {
			$subtotal += WC()->cart->get_subtotal_tax();
		}

		if ( $amount > 0 ) {

			$perc_discount  = round( $amount / $subtotal, 2 ) * 100;
			$price_discount = $subtotal - $amount;
			$message        = str_replace( '%total_discount_percentage%', $perc_discount . '%', $original_message );
			$message        = str_replace( '%total_discount_price%', wc_price( $price_discount ), $message );
		}

		return $message;
	}
}

if ( ! function_exists( 'yith_ywdpd_filter_onstock_products' ) ) {
	/**
	 * Return only product ids that are onstock
	 *
	 * @param array $product_ids The product ids.
	 *
	 * @return array
	 */
	function yith_ywdpd_filter_onstock_products( $product_ids ) {
		global $wpdb;

		$query   = $wpdb->prepare( "SELECT product_id FROM {$wpdb->prefix}wc_product_meta_lookup WHERE stock_status=%s;", 'outofstock' );
		$results = $wpdb->get_results( $query, ARRAY_A );

		if ( $results ) {
            $results = wp_list_pluck( $results, 'product_id' );
			foreach ( $product_ids as $index => $product_id ) {
				if ( in_array( $product_id, $results, true ) ) {
					unset( $product_ids[ $index ] );
				}
			}
		}
        return $product_ids;
	}
}