<?php
/**
 * Compatibility with YITH WooCommerce Brands Add-on Premium
 *
 * @package YITH WooCommerce Dynamic Pricing & Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH <plugins@yithemes.com>
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * YWDPD_Brands class to add compatibility with YITH WooCommerce Brands Add-on Premium
 *
 * @class   YWDPD_Brands
 * @package YITH WooCommerce Dynamic Pricing & Discounts
 * @since   1.1.7
 */
if ( ! class_exists( 'YITH_YWDPD_Brands_Addon_Integration' ) ) {

	/**
	 * Class YITH_YWDPD_Brands_Addon_Integration
	 */
	class YITH_YWDPD_Brands_Addon_Integration {
		/**
		 * Constructor
		 *
		 * Initialize class and registers actions and filters to be used
		 *
		 * @since  1.1.7
		 */
		public function __construct() {

			add_filter( 'ywdpd_custom_taxonomy', array( $this, 'add_brands_taxonomy_in_counters' ), 10, 1 );

			add_filter( 'ywdpd_get_rule_for_options', array( $this, 'add_pricing_rule_option_rule_for' ) );
			add_filter( 'ywdpd_get_exclude_rule_for_options', array( $this, 'add_pricing_rule_option_rule_for' ) );
			add_filter( 'ywdpd_get_adjustment_to_options', array( $this, 'add_pricing_rule_option_rule_for' ) );
			add_filter( 'ywdpd_get_exclude_adjustment_to_options', array( $this, 'add_pricing_rule_option_rule_for' ) );
			add_filter( 'ywdpd_get_rule_for_search_fields', array( $this, 'add_pricing_rule_option' ) );
			add_filter(
				'ywdpd_get_exclude_rule_for_search_fields',
				array(
					$this,
					'add_pricing_rule_option',
				)
			);
			add_filter( 'ywdpd_get_adjustment_to_search_fields', array( $this, 'add_pricing_rule_option' ) );
			add_filter(
				'ywdpd_get_exclude_adjustment_to_search_fields',
				array(
					$this,
					'add_pricing_rule_option',
				)
			);

			// Valid price rule.

			add_filter( 'ywdpd_is_valid_to_apply', array( $this, 'valid_rule_for_brands' ), 21, 4 );
			add_filter( 'ywdpd_is_product_excluded_from_apply', array( $this, 'product_excluded_for_brands' ), 20, 4 );
			add_filter( 'ywdpd_get_comulative_quantity', array( $this, 'brands_comulative_quantity' ), 20, 3 );
			add_filter( 'ywdpd_is_valid_to_adjust', array( $this, 'valid_rule_adjustment_to_brands' ), 20, 4 );

			// CART RULES.
			add_filter(
				'ywdpd_advanced_conditions_product_include_fields',
				array(
					$this,
					'add_advanced_condition_option',
				)
			);
			add_filter(
				'ywdpd_advanced_conditions_product_exclude_fields',
				array(
					$this,
					'add_advanced_condition_option',
				)
			);
			add_filter(
				'ywdpd_advanced_conditions_product_disable_fields',
				array(
					$this,
					'add_advanced_condition_option',
				)
			);

			add_filter( 'ywdpd_valid_product_cart_condition', array( $this, 'valid_product_cart_condition' ), 15, 4 );
			add_filter(
				'ywpd_is_excluded_product_cart_condition',
				array(
					$this,
					'valid_product_exclude_cart_condition',
				),
				15,
				3
			);

			// SPECIAL OFFER IN POPUP.
			add_filter( 'ywdpd_involved_items', array( $this, 'add_brands_args_in_item' ), 20, 3 );
			add_filter(
				'ywdpd_involved_adjustment_items',
				array(
					$this,
					'add_brands_adjustment_args_in_item',
				),
				20,
				3
			);
			add_filter(
				'ywdpd_get_product_ids_involved_to_exclude',
				array(
					$this,
					'return_product_ids_to_exclude',
				),
				20,
				3
			);
			add_filter(
				'ywdpd_get_product_ids_involved_adjustment_to_exclude',
				array(
					$this,
					'return_product_ids_to_exclude',
				),
				20,
				3
			);

			add_filter( 'ywdpd_product_ids_included_data_store', array( $this, 'add_product_in_data_store' ), 10, 2 );
			add_filter( 'ywdpd_get_scheduled_ids', array( $this, 'add_product_in_data_store' ), 10, 2 );
			add_filter(
				'ywdpd_product_ids_excluded_data_store',
				array(
					$this,
					'add_excluded_product_in_data_store',
				),
				10,
				2
			);
			add_filter(
				'ywdpd_get_scheduled_excluded_ids',
				array(
					$this,
					'add_excluded_product_in_data_store',
				),
				10,
				2
			);
			add_filter(
				'ywdpd_can_process_schedule_in_edit_product',
				array(
					$this,
					'can_process_schedule_in_edit_product',
				),
				10,
				2
			);
		}

		/**
		 * Add the vendor taxonomy in counter
		 *
		 * @param array $custom_taxonomy The custom taxonomy.
		 *
		 * @return array
		 * @since  3.0.0
		 */
		public function add_brands_taxonomy_in_counters( $custom_taxonomy ) {

			$custom_taxonomy[] = YITH_WCBR::$brands_taxonomy;

			return $custom_taxonomy;
		}

		/**
		 * Check if the rule is valid
		 *
		 * @param bool             $is_valid If rule is valid.
		 * @param string           $type     The rule product type.
		 * @param WC_Product       $product  The product.
		 * @param YWDPD_Price_Rule $rule     The rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function valid_rule_for_brands( $is_valid, $type, $product, $rule ) {

			if ( ! $is_valid ) {
				if ( 'specific_brands' === $type ) {

					$brand_list_ids = $rule->get_meta( 'apply_to_brands_list' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_valid = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
			}

			return $is_valid;
		}

		/**
		 * Check if this product is excluded from a rule
		 *
		 * @param bool             $is_excluded If is excluded.
		 * @param string           $type        The rule product type.
		 * @param WC_Product       $product     The product.
		 * @param YWDPD_Price_Rule $rule        The rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function product_excluded_for_brands( $is_excluded, $type, $product, $rule ) {

			if ( 'brand_list_excluded' === $type && $rule->is_exclude_from_apply_enabled() ) {
				$brand_list_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
				}
				$is_excluded = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
			}

			return $is_excluded;
		}

		/**
		 * Get the comulative quantity for brands
		 *
		 * @param int              $quantity The quantity.
		 * @param string           $rule_for The rule type.
		 * @param YWDPD_Price_Rule $rule     The rule.
		 *
		 * @return int
		 * @since  3.0.0
		 */
		public function brands_comulative_quantity( $quantity, $rule_for, $rule ) {
			$is_exclude_activated = $rule->is_exclude_from_apply_enabled();
			$type                 = $rule->get_rule_for();

			if ( 'specific_brands' === $type ) {
				$brand_list_ids = $rule->get_meta( 'apply_to_brands_list' );
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
				}
				$quantity = YWDPD_Counter::get_comulative_taxonomy_count( $brand_list_ids, YITH_WCBR::$brands_taxonomy );

			}

			if ( $is_exclude_activated ) {
				$ex_type = $rule->get_exclude_rule_for();
				if ( 'brand_list_excluded' === $ex_type ) {
					$brand_list_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$qty_excl = YWDPD_Counter::get_comulative_taxonomy_count( $brand_list_ids, YITH_WCBR::$brands_taxonomy );

					$quantity -= $qty_excl;
				}
			}

			return $quantity;
		}

		/**
		 * Check if the rule is valid to adjustment to
		 *
		 * @param bool             $is_valid Is valid.
		 * @param string           $type     The check type.
		 * @param WC_Product       $product  The product.
		 * @param YWDPD_Price_Rule $rule     The price rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function valid_rule_adjustment_to_brands( $is_valid, $type, $product, $rule ) {
			$is_excluded      = false;
			$is_active_adjust = $rule->is_enabled_apply_adjustment_to();
			if ( $is_active_adjust ) {

				if ( $rule->is_exclude_adjustment_to_enabled() && 'brand_list_excluded' === $rule->get_exclude_apply_adjustment_rule_for() ) {
					$brand_list_ids = $rule->get_meta( 'apply_adjustment_brands_list_excluded' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_excluded = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
				if ( ! $is_excluded && 'specific_brands' === $rule->get_rule_apply_adjustment_discount_for() ) {
					$brand_list_ids = $rule->get_meta( 'apply_adjustment_brands_list' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_valid = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
			}

			return $is_valid;
		}

		/**
		 * Add pricing rules options in settings panels
		 *
		 * @param array $rules The cart rule options.
		 *
		 * @return array
		 */
		public function add_advanced_condition_option( $rules ) {

			if ( 'ywdpd_advanced_conditions_product_include_fields' === current_filter() ) {

				$options = array(
					array(
						'id'        => 'enable_require_product_brands',
						'name'      => __( 'Require specific brands in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to require products of specific brands in the cart to apply the discount', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product require_product',
						'class'     => 'ywdpd_enable_require_product_brands',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'require_product',
									),
								)
							),
						),
					),
					array(
						'id'        => 'require_product_brands_list',
						'name'      => __( 'Include a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands are required in cart to apply the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'             => YITH_WCBR::$brands_taxonomy,
							'placeholder'          => __( 'Search for a brand', 'ywdpd' ),
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'require_product',
									),
									array(
										'id'    => 'ywdpd_enable_require_product_brands',
										'value' => 'yes',
									),
								)
							),
						),
						'class_row' => 'product require_product enable_require_product_brands_list',
						'multiple'  => true,
					),
					array(
						'id'        => 'require_product_brand_list_mode',
						'name'      => __( 'Apply the discount if:', 'ywdpd' ),
						'desc'      => __( 'Choose whether to apply the discount when at least one of the specified product brands is in the cart or only when all of them are in the cart', 'ywdpd' ),
						'type'      => 'radio',
						'options'   => array(
							'at_least'  => __( 'At least one selected brand is in cart', 'ywdpd' ),
							'all_brand' => __( 'All selected brands are in cart', 'ywdpd' ),
						),
						'default'   => 'at_least',
						'class_row' => 'product require_product enable_require_product_brands_list',
						'class'     => 'ywdpd_require_product_tag_list_mode',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'require_product',
									),
									array(
										'id'    => 'ywdpd_enable_require_product_brands',
										'value' => 'yes',
									),
								)
							),
						),
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_advanced_conditions_product_exclude_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_exclude_product_brands',
						'name'      => __( 'Exclude specific brands from discount validation', 'ywdpd' ),
						'desc'      => __( 'Enable if you want to exclude specific brands from this cart discount validation', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product exclude_product',
						'class'     => 'ywdpd_enable_exclude_product_brands',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'exclude_product',
									),
								)
							),
						),
					),
					array(
						'id'        => 'exclude_product_brands_list',
						'name'      => __( 'Exclude a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands to exclude from this cart discount validation', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'             => YITH_WCBR::$brands_taxonomy,
							'placeholder'          => __( 'Search for a brand', 'ywdpd' ),
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'exclude_product',
									),
									array(
										'id'    => 'ywdpd_enable_exclude_product_brands',
										'value' => 'yes',
									),
								)
							),

						),
						'class_row' => 'product exclude_product enable_exclude_product_brands_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_advanced_conditions_product_disable_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_disable_product_brands',
						'name'      => __( 'Disable the discount when there are specific brands in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to disable the discount if users have specific brands in their cart', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product disable_product',
						'class'     => 'ywdpd_enable_disable_product_brands',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'disable_product',
									),
								)
							),
						),
					),
					array(
						'id'        => 'disable_product_brands_list',
						'name'      => __( 'Select a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands will disable the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'             => YITH_WCBR::$brands_taxonomy,
							'placeholder'          => __( 'Search for a brand', 'ywdpd' ),
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'disable_product',
									),
									array(
										'id'    => 'ywdpd_enable_disable_product_brands',
										'value' => 'yes',
									),
								),
							),
						),
						'class_row' => 'product disable_product enable_disable_product_brands_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			}

			return $rules;
		}

		/**
		 * Valid the cart rule for vendors
		 *
		 * @param bool            $is_valid   If rule is valid.
		 * @param array           $condition  The specific condition.
		 * @param array           $conditions All conditions.
		 * @param YWDPD_Cart_Rule $rule       The cart rule.
		 *
		 * @return bool
		 */
		public function valid_product_cart_condition( $is_valid, $condition, $conditions, $rule ) {

			$type_check = $rule->get_condition_product_type( $condition );

			if ( 'require_product' === $type_check ) {
				$require_product_brands = yith_plugin_fw_is_true( $condition['enable_require_product_brands'] );
				$brands_to_check        = $condition['require_product_brands_list'] ?? array();
				$brand_check_mode       = $condition['require_product_brand_list_mode'];

				if ( $require_product_brands && is_array( $brands_to_check ) && count( $brands_to_check ) > 0 ) {
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brands_to_check = $this->get_translated_brand_ids( $brands_to_check );
					}
					$is_valid = YWDPD_Counter::check_taxonomy_in_cart( $brands_to_check, YITH_WCBR::$brands_taxonomy, $brand_check_mode );
				}
			} elseif ( 'disable_product' === $type_check ) {
				$disable_brand     = isset( $condition['enable_disable_product_brands'] ) && yith_plugin_fw_is_true( $condition['enable_disable_product_brands'] );
				$brands_to_disable = $condition['disable_product_brands_list'] ?? array();

				if ( $disable_brand && count( $brands_to_disable ) > 0 ) {
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brands_to_disable = $this->get_translated_brand_ids( $brands_to_disable );
					}

					$is_valid = ! YWDPD_Counter::check_taxonomy_in_cart( $brands_to_disable, YITH_WCBR::$brands_taxonomy, 'at_least' );
				}
			}

			return $is_valid;
		}


		/**
		 * Check if brands are  excluded
		 *
		 * @param bool       $is_excluded If the rule is excluded.
		 * @param array      $condition   The condition.
		 * @param WC_Product $product     The product.
		 *
		 * @return bool
		 */
		public function valid_product_exclude_cart_condition( $is_excluded, $condition, $product ) {

			$exclude_product_brand = $condition['enable_exclude_product_brands'] ?? false;
			$brands_excluded_list  = $condition['exclude_product_brands_list'] ?? array();
			if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
				$brands_excluded_list = $this->get_translated_brand_ids( $brands_excluded_list );
			}
			if ( ! $is_excluded ) {
				if ( yith_plugin_fw_is_true( $exclude_product_brand ) && count( $brands_excluded_list ) > 0 ) {

					$product_terms = YWDPD_Taxonomy::get_product_term_ids( $product, YITH_WCBR::$brands_taxonomy );

					$is_excluded = count( array_intersect( $product_terms, $brands_excluded_list ) ) > 0;

				}
			}

			return $is_excluded;
		}


		/**
		 * Add pricing rules options in settings panels
		 *
		 * @param array $rules The pricing rule options.
		 *
		 * @return array
		 */
		public function add_pricing_rule_option_rule_for( $rules ) {
			$filter = current_filter();
			$id     = 'ywdpd_get_rule_for_options' === $filter || 'ywdpd_get_adjustment_to_options' === $filter ? 'specific_brands' : 'brand_list_excluded';

			$new_rule = array(
				$id => __( 'Specific product brands', 'ywdpd' ),
			);

			return array_merge( $rules, $new_rule );
		}

		/**
		 * Add brand search field in the price rule metabox
		 *
		 * @param array $fields The fields.
		 *
		 * @return array
		 * @since  2.0.0
		 */
		public function add_pricing_rule_option( $fields ) {
			$current_filter = current_filter();
			if ( 'ywdpd_get_rule_for_search_fields' === $current_filter ) {
				$option = $this->get_brand_search_field(
					'apply_to_brands_list',
					__( 'Apply rule to:', 'ywdpd' ),
					__( 'Search the brand(s) to include in the rule', 'ywdpd' ),
					true,
					array(
						array(
							'id'    => '_rule_for',
							'value' => 'specific_brands',
						),
					)
				);
			} elseif ( 'ywdpd_get_exclude_rule_for_search_fields' === $current_filter ) {
				$option = $this->get_brand_search_field(
					'apply_to_brands_list_excluded',
					__( 'Choose which brand(s) to exclude', 'ywdpd' ),
					__( 'Search the brand(s) to exclude from the rule', 'ywdpd' ),
					true,
					array(
						array(
							'id'    => '_active_exclude',
							'value' => 'yes',
						),
						array(
							'id'      => '_rule_for',
							'value'   => 'specific_products',
							'compare' => '!=',
						),
						array(
							'id'    => '_exclude_rule_for',
							'value' => 'brand_list_excluded',
						),
					)
				);
			} elseif ( 'ywdpd_get_adjustment_to_search_fields' === $current_filter ) {
				$option = $this->get_brand_search_field(
					'apply_adjustment_brands_list',
					__( 'Apply rule to:', 'ywdpd' ),
					__( 'Search the brand(s) to include in the rule', 'ywdpd' ),
					true,
					array(
						array(
							'id'    => '_rule_apply_adjustment_discount_for',
							'value' => 'specific_brands',
						),
						array(
							'id'    => '_active_apply_discount_to',
							'value' => 'yes',
						),
					)
				);
			} else {
				$option = $this->get_brand_search_field(
					'apply_adjustment_brands_list_excluded',
					__( 'Apply rule to:', 'ywdpd' ),
					__( 'Search the brand(s) to include in the rule', 'ywdpd' ),
					true,
					array(
						array(
							'id'    => '_exclude_apply_adjustment_rule_for',
							'value' => 'brand_list_excluded',
						),
						array(
							'id'    => '_active_apply_adjustment_to_exclude',
							'value' => 'yes',
						),
						array(
							'id'      => '_rule_apply_adjustment_discount_for',
							'value'   => 'specific_products',
							'compare' => '!=',
						),
						array(
							'id'    => '_active_apply_discount_to',
							'value' => 'yes',
						),
					)
				);
			}

			return array_merge( $fields, $option );

		}

		/**
		 * Add the brands option in pricing rule metabox
		 *
		 * @param array $pricing_options The options.
		 *
		 * @return array
		 */
		public function add_brands_pricing_options( $pricing_options ) {

			$start        = $pricing_options['tabs']['settings']['fields'];
			$position     = array_search( 'apply_to_tags_list_excluded', array_keys( $start ), true );
			$begin        = array_slice( $start, 0, $position + 1 );
			$end          = array_slice( $start, $position );
			$brands_items = array(
				'apply_to_brands_list'          => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_to',
						'values' => 'brand_list',
					),
				),
				'apply_to_brands_list_excluded' => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_to',
						'values' => 'brand_list_excluded',
					),
				),
			);

			$start        = $begin + $brands_items + $end;
			$position     = array_search( 'apply_adjustment_tags_list', array_keys( $start ), true );
			$begin        = array_slice( $start, 0, $position + 1 );
			$end          = array_slice( $start, $position );
			$brands_items = array(
				'apply_adjustment_brands_list'          => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_adjustment',
						'values' => 'brand_list',
					),
				),
				'apply_adjustment_brands_list_excluded' => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_adjustment',
						'values' => 'brand_list_excluded',
					),
				),
			);

			$pricing_options['tabs']['settings']['fields'] = $begin + $brands_items + $end;

			return $pricing_options;

		}

		/**
		 * Add the brand taxonomy in popup.
		 *
		 * @param array            $items      The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for   The rule for.
		 *
		 * @return array
		 */
		public function add_brands_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'specific_brands' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_to_brands_list' ),
					'type'     => YITH_WCBR::$brands_taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Add the brands taxonomy in popup.
		 *
		 * @param array            $items      The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for   The rule for.
		 *
		 * @return array
		 */
		public function add_brands_adjustment_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'specific_brands' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_adjustment_brands_list' ),
					'type'     => YITH_WCBR::$brands_taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Return the product ids for specific brands
		 *
		 * @param array            $product_ids Product ids.
		 * @param YWDPD_Price_Rule $rule        The rule.
		 * @param string           $exclude_for The type of exclusion.
		 *
		 * @return array
		 * @since  2.0.0
		 */
		public function return_product_ids_to_exclude( $product_ids, $rule, $exclude_for ) {

			if ( 'brand_list_excluded' === $exclude_for ) {
				if ( 'ywdpd_get_product_ids_involved_to_exclude' === current_filter() ) {
					$brand_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
				} else {
					$brand_ids = $rule->get_meta( 'apply_adjustment_brands_list_excluded' );
				}
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_ids = $this->get_translated_brand_ids( $brand_ids );
				}
				$product_ids = $this->get_product_ids_by_brands( $brand_ids );
			}

			return $product_ids;
		}

		/**
		 * Return all product by brand
		 *
		 * @param int[] $brand_ids The brand ids.
		 *
		 * @return int[]
		 */
		public function get_product_ids_by_brands( $brand_ids ) {

			if ( is_array( $brand_ids ) && count( $brand_ids ) > 0 ) {
				$tax_query = WC()->query->get_tax_query(
					array(
						array(
							'taxonomy' => YITH_WCBR::$brands_taxonomy,
							'terms'    => array_values( $brand_ids ),
							'operator' => 'IN',
						),
					)
				);

				return get_posts(
					array(
						'numberposts' => - 1,
						'post_type'   => array( 'product' ),
						'post_status' => 'publish',
						'tax_query'   => $tax_query, // phpcs:ignore WordPress.DB.SlowDBQuery
						'fields'      => 'ids',
					)
				);
			}

			return array();
		}

		/**
		 * Add in the array the translated brand ids.
		 *
		 * @param array $brand_ids The original brand ids.
		 *
		 * @return array
		 * @since  3.0.0
		 */
		public function get_translated_brand_ids( $brand_ids ) {
			$translated_ids = array();

			foreach ( $brand_ids as $brand_id ) {

				$translated_id = intval( yit_wpml_object_id( $brand_id, YITH_WCBR::$brands_taxonomy, true ) );

				if ( intval( $brand_id ) !== $translated_id && ! in_array( $translated_id, $translated_ids, true ) ) {
					$translated_ids[] = $translated_id;
				}
			}

			return count( $translated_ids ) > 0 ? $translated_ids : $brand_ids;
		}

		/**
		 * Return the search field configuration
		 *
		 * @param string $id       The field id.
		 * @param string $label    The label.
		 * @param string $desc     The description.
		 * @param bool   $multiple Is multiple or not.
		 * @param array  $deps     The deps.
		 *
		 * @return array[]
		 */
		public function get_brand_search_field( $id, $label, $desc, $multiple, $deps ) {
			return array(
				$id => array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_WCBR::$brands_taxonomy,
						'placeholder' => __( 'Search for a brand', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							$deps,
						),
					),
					'label'    => $label,
					'desc'     => $desc,
					'multiple' => $multiple,
					'required' => true,
					'std'      => $multiple ? array() : '',
				),
			);
		}

		/**
		 * Add the product id by vendor
		 *
		 * @param int[]            $product_ids The ids list.
		 * @param YWDPD_Price_Rule $rule        The rule.
		 *
		 * @return int[]
		 */
		public function add_product_in_data_store( $product_ids, $rule ) {
			if ( 'specific_brands' === $rule->get_rule_for() ) {
				$brand_ids   = $rule->get_meta( 'apply_to_brands_list' ) ?? array();
				$list_ids    = $this->get_product_ids_by_brands( $brand_ids );
				$product_ids = array_merge( $product_ids, $list_ids );
			}

			return $product_ids;
		}

		/**
		 * Add the product id by vendor
		 *
		 * @param int[]            $excluded_product_ids The ids list.
		 * @param YWDPD_Price_Rule $rule                 The rule.
		 *
		 * @return int[]
		 */
		public function add_excluded_product_in_data_store( $excluded_product_ids, $rule ) {
			if ( $rule->is_exclude_from_apply_enabled() ) {
				if ( 'brand_list_excluded' === $rule->get_exclude_rule_for() ) {
					$brand_ids            = $rule->get_meta( 'apply_to_brands_list_excluded' ) ?? array();
					$list_ids             = $this->get_product_ids_by_brands( $brand_ids );
					$excluded_product_ids = array_merge( $excluded_product_ids, $list_ids );
				}
			}

			return $excluded_product_ids;
		}

		/**
		 * Check if the product change the brands
		 *
		 * @param bool       $has_changes The product has changes.
		 * @param WC_Product $product     The product.
		 *
		 * @return bool
		 */
		public function can_process_schedule_in_edit_product( $has_changes, $product ) {
			if ( ! $has_changes ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
				$post_brands_slug = isset( $_POST['tax_input']['yith_product_brand'] ) ? array_filter( wp_unslash( $_POST['tax_input']['yith_product_brand'] ) ) : array();
				$brands           = wp_get_post_terms( $product->get_id(), YITH_WCBR::$brands_taxonomy, array( 'fields' => 'ids' ) );
				$check            = array_diff( array_unique( array_merge( $brands, $post_brands_slug ) ), array_intersect( $brands, $post_brands_slug ) );
				$has_changes      = count( $check ) > 0;
			}

			return $has_changes;
		}
	}
}
