<?php
/**
 * Compatibility with YITH WooCommerce Multivendor
 *
 * @package YITH WooCommerce Dynamic Pricing & Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH <plugins@yithemes.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * YWDPD_Multivendor class to add compatibility with YITH WooCommerce Multivendor
 *
 * @class   YWDPD_Multivendor
 * @package YITH WooCommerce Dynamic Pricing & Discounts
 * @since   1.0.0
 */
if ( ! class_exists( 'YITH_YWDPD_Multi_Vendor_Integration' ) ) {

	/**
	 * Class YITH_YWDPD_Multi_Vendor_Integration
	 */
	class YITH_YWDPD_Multi_Vendor_Integration {
		/**
		 * Constructor
		 *
		 * Initialize class and registers actions and filters to be used
		 *
		 * @since  1.0.0
		 */
		public function __construct() {
			add_filter( 'ywdpd_get_rule_for_options', array( $this, 'add_vendor_rule_for_option' ) );
			add_filter( 'ywdpd_get_exclude_rule_for_options', array( $this, 'add_vendor_rule_for_option' ) );
			add_filter( 'ywdpd_get_adjustment_to_options', array( $this, 'add_vendor_rule_for_option' ) );
			add_filter( 'ywdpd_get_exclude_adjustment_to_options', array( $this, 'add_vendor_rule_for_option' ) );
			add_filter( 'ywdpd_get_rule_for_search_fields', array( $this, 'add_vendor_rule_for_search_field' ) );
			add_filter(
				'ywdpd_get_exclude_rule_for_search_fields',
				array(
					$this,
					'add_vendor_exclude_rule_for_search_field',
				)
			);
			add_filter( 'ywdpd_get_adjustment_to_search_fields', array( $this, 'add_vendor_adjustment_search_field' ) );
			add_filter(
				'ywdpd_get_exclude_adjustment_to_search_fields',
				array(
					$this,
					'add_vendor_exclude_adjustment_search_field',
				)
			);
			add_filter( 'ywdpd_is_valid_to_apply', array( $this, 'valid_rule_for_vendor' ), 20, 4 );
			add_filter( 'ywdpd_is_product_excluded_from_apply', array( $this, 'product_excluded_for_vendor' ), 20, 4 );
			add_filter( 'ywdpd_is_valid_to_adjust', array( $this, 'valid_rule_adjustment_to_vendor' ), 20, 4 );
			add_filter( 'ywdpd_custom_taxonomy', array( $this, 'add_vendor_taxonomy_in_counters' ), 10, 1 );

			// SPECIAL OFFER IN POPUP.

			add_filter( 'ywdpd_involved_items', array( $this, 'add_vendor_args_in_item' ), 20, 3 );
			add_filter(
				'ywdpd_involved_adjustment_items',
				array(
					$this,
					'add_vendor_adjustment_args_in_item',
				),
				20,
				3
			);
			add_filter(
				'ywdpd_get_product_ids_involved_to_exclude',
				array(
					$this,
					'return_product_ids_to_exclude',
				),
				20,
				3
			);
			add_filter(
				'ywdpd_get_product_ids_involved_adjustment_to_exclude',
				array(
					$this,
					'return_product_ids_to_exclude',
				),
				20,
				3
			);
			add_filter( 'ywdpd_get_comulative_quantity', array( $this, 'check_comulative_quantity' ), 20, 3 );

			// Advanced conditions.
			add_filter(
				'ywdpd_advanced_conditions_product_include_fields',
				array(
					$this,
					'add_advanced_condition_option',
				)
			);
			add_filter(
				'ywdpd_advanced_conditions_product_exclude_fields',
				array(
					$this,
					'add_advanced_condition_option',
				)
			);
			add_filter(
				'ywdpd_valid_product_cart_condition',
				array(
					$this,
					'yith_vendor_valid_require_vendor_condition',
				),
				20,
				4
			);
			add_filter(
				'ywpd_is_excluded_product_cart_condition',
				array(
					$this,
					'yith_vendor_valid_exclude_vendor_condition',
				),
				20,
				4
			);

			add_filter( 'ywdpd_product_ids_included_data_store', array( $this, 'add_product_in_data_store' ), 10, 2 );
			add_filter( 'ywdpd_get_scheduled_ids', array( $this, 'add_product_in_data_store' ), 10, 2 );
			add_filter(
				'ywdpd_product_ids_excluded_data_store',
				array(
					$this,
					'add_excluded_product_in_data_store',
				),
				10,
				2
			);
			add_filter(
				'ywdpd_get_scheduled_excluded_ids',
				array(
					$this,
					'add_excluded_product_in_data_store',
				),
				10,
				2
			);
			add_filter(
				'ywdpd_can_process_schedule_in_edit_product',
				array(
					$this,
					'can_process_schedule_in_edit_product',
				),
				10,
				2
			);
		}

		/**
		 * Add the option for vendors
		 *
		 * @param array $rule_for_options The options.
		 *
		 * @return array
		 */
		public function add_vendor_rule_for_option( $rule_for_options ) {
			$id             = 'ywdpd_get_exclude_adjustment_to_options' === current_filter() ? 'vendor_list_excluded' : 'vendor_list';
			$vendor_options = array(
				$id => __( 'Specific vendors', 'ywdpd' ),
			);

			return array_merge( $rule_for_options, $vendor_options );
		}

		/**
		 * Add the search for vendor field
		 *
		 * @param array $rule_for_search_fields The search fields.
		 *
		 * @return array
		 */
		public function add_vendor_rule_for_search_field( $rule_for_search_fields ) {
			$vendor_search_option = $this->get_vendor_search_field(
				'apply_to_vendors_list',
				__( 'Apply rule to:', 'ywdpd' ),
				__( 'Search the vendor(s) to include in the rule', 'ywdpd' ),
				true,
				array(
					array(
						'id'    => '_rule_for',
						'value' => 'vendor_list',
					),
				)
			);

			return array_merge( $rule_for_search_fields, $vendor_search_option );
		}

		/**
		 * Add the exclude search for vendor field
		 *
		 * @param array $exclude_rule_for_search_fields The search fields.
		 *
		 * @return array
		 */
		public function add_vendor_exclude_rule_for_search_field( $exclude_rule_for_search_fields ) {
			$vendor_search_option = $this->get_vendor_search_field(
				'apply_to_vendors_list_excluded',
				__( 'Choose which vendor(s) to exclude', 'ywdpd' ),
				__( 'Search the vendor(s) to exclude from this discount', 'ywdpd' ),
				true,
				array(
					array(
						'id'    => '_active_exclude',
						'value' => 'yes',
					),
					array(
						'id'      => '_rule_for',
						'value'   => 'specific_products',
						'compare' => '!=',
					),
					array(
						'id'    => '_exclude_rule_for',
						'value' => 'vendor_list',
					),
				)
			);

			return array_merge( $exclude_rule_for_search_fields, $vendor_search_option );
		}

		/**
		 * Add the search for vendor field
		 *
		 * @param array $adjustment_to_search_field The search fields.
		 *
		 * @return array
		 */
		public function add_vendor_adjustment_search_field( $adjustment_to_search_field ) {
			$vendor_search_option = $this->get_vendor_search_field(
				'apply_adjustment_vendor_list',
				__( 'Choose which vendor(s) to include', 'ywdpd' ),
				__( 'Search the vendor(s) to include in this discount', 'ywdpd' ),
				true,
				array(
					array(
						'id'    => '_rule_apply_adjustment_discount_for',
						'value' => 'vendor_list',
					),
					array(
						'id'    => '_active_apply_discount_to',
						'value' => 'yes',
					),
				)
			);

			return array_merge( $adjustment_to_search_field, $vendor_search_option );
		}

		/**
		 * Add the search for vendor field
		 *
		 * @param array $exclude_adjustment_to_search_field The search fields.
		 *
		 * @return array
		 */
		public function add_vendor_exclude_adjustment_search_field( $exclude_adjustment_to_search_field ) {
			$vendor_search_option = $this->get_vendor_search_field(
				'apply_adjustment_vendor_list_excluded',
				__( 'Choose which vendor(s) to exclude', 'ywdpd' ),
				__( 'Search the vendor(s) to exclude from this discount', 'ywdpd' ),
				true,
				array(
					array(
						'id'    => '_exclude_apply_adjustment_rule_for',
						'value' => 'vendor_list_excluded',
					),
					array(
						'id'    => '_active_apply_adjustment_to_exclude',
						'value' => 'yes',
					),
					array(
						'id'      => '_rule_apply_adjustment_discount_for',
						'value'   => 'specific_products',
						'compare' => '!=',
					),
					array(
						'id'    => '_active_apply_discount_to',
						'value' => 'yes',
					),
				)
			);

			return array_merge( $exclude_adjustment_to_search_field, $vendor_search_option );
		}

		/**
		 * Check if the rule is valid
		 *
		 * @param bool             $is_valid If rule is valid.
		 * @param string           $type     The rule product type.
		 * @param WC_Product       $product  The product.
		 * @param YWDPD_Price_Rule $rule     The rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function valid_rule_for_vendor( $is_valid, $type, $product, $rule ) {

			if ( 'vendor_list' === $type ) {

				$vendor_list_ids = $rule->get_meta( 'apply_to_vendors_list' );
				$is_valid        = $rule->is_product_in_list( $product, $vendor_list_ids, $this->get_vendor_taxonomy_name() );
			}

			return $is_valid;
		}

		/**
		 * Check if this product is excluded from a rule
		 *
		 * @param bool             $is_excluded If is excluded.
		 * @param string           $type        The rule product type.
		 * @param WC_Product       $product     The product.
		 * @param YWDPD_Price_Rule $rule        The rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function product_excluded_for_vendor( $is_excluded, $type, $product, $rule ) {

			if ( 'vendor_list_excluded' === $type && $rule->is_exclude_from_apply_enabled() ) {
				$vendor_list_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
				$is_excluded     = $rule->is_product_in_list( $product, $vendor_list_ids, $this->get_vendor_taxonomy_name() );
			}

			return $is_excluded;
		}

		/**
		 * Check if the rule is valid to adjustment to
		 *
		 * @param bool             $is_valid Is valid.
		 * @param string           $type     The check type.
		 * @param WC_Product       $product  The product.
		 * @param YWDPD_Price_Rule $rule     The price rule.
		 *
		 * @return bool
		 * @since  3.0.0
		 */
		public function valid_rule_adjustment_to_vendor( $is_valid, $type, $product, $rule ) {
			$is_excluded      = false;
			$is_active_adjust = $rule->is_enabled_apply_adjustment_to();
			if ( $is_active_adjust ) {

				if ( $rule->is_exclude_adjustment_to_enabled() && 'vendor_list_excluded' === $rule->get_exclude_apply_adjustment_rule_for() ) {
					$vendor_list_ids = $rule->get_meta( 'apply_adjustment_vendor_list_excluded' );
					$is_excluded     = $rule->is_product_in_list( $product, $vendor_list_ids, $this->get_vendor_taxonomy_name() );
				}
				if ( ! $is_excluded && 'vendor_list' === $rule->get_rule_apply_adjustment_discount_for() ) {
					$vendor_list_ids = $rule->get_meta( 'apply_adjustment_vendor_list' );
					$is_valid        = $rule->is_product_in_list( $product, $vendor_list_ids, $this->get_vendor_taxonomy_name() );
				}
			}

			return $is_valid;
		}

		/**
		 * Return the vendor taxonomy name
		 *
		 * @return string
		 * @since  1.13.0
		 */
		public function get_vendor_taxonomy_name() {
			$taxonomy_name = class_exists( 'YITH_Vendors_Taxonomy' ) ? YITH_Vendors_Taxonomy::TAXONOMY_NAME : YITH_Vendors()->get_taxonomy_name();

			return $taxonomy_name;
		}

		/**
		 * Return the search field configuration
		 *
		 * @param string $id       The field id.
		 * @param string $label    The label.
		 * @param string $desc     The description.
		 * @param bool   $multiple Is multiple or not.
		 * @param array  $deps     The deps.
		 *
		 * @return array[]
		 */
		public function get_vendor_search_field( $id, $label, $desc, $multiple, $deps ) {
			return array(
				$id => array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => $this->get_vendor_taxonomy_name(),
						'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							$deps,
						),
					),
					'label'    => $label,
					'desc'     => $desc,
					'multiple' => $multiple,
					'required' => true,
					'std'      => $multiple ? array() : '',
				),
			);
		}

		/**
		 * Add the vendor taxonomy in counter
		 *
		 * @param array $custom_taxonomy The custom taxonomy.
		 *
		 * @return array
		 * @since  3.0.0
		 */
		public function add_vendor_taxonomy_in_counters( $custom_taxonomy ) {

			$custom_taxonomy[] = $this->get_vendor_taxonomy_name();

			return $custom_taxonomy;
		}

		/**
		 * Add the vendor taxonomy in popup.
		 *
		 * @param array            $items      The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for   The rule for.
		 *
		 * @return array
		 */
		public function add_vendor_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'vendor_list' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_to_vendors_list' ),
					'type'     => $this->get_vendor_taxonomy_name(),
				);
			}

			return $items;
		}

		/**
		 * Add the vendor taxonomy in popup.
		 *
		 * @param array            $items      The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for   The rule for.
		 *
		 * @return array
		 */
		public function add_vendor_adjustment_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'vendor_list' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_adjustment_vendor_list' ),
					'type'     => $this->get_vendor_taxonomy_name(),
				);
			}

			return $items;
		}

		/**
		 * Return the product id to exclude
		 *
		 * @param array            $product_ids The product ids.
		 * @param YWDPD_Price_Rule $rule        The cart rule.
		 * @param string           $exclude_for The exclusion type.
		 *
		 * @return array
		 * @since  3.0.0
		 */
		public function return_product_ids_to_exclude( $product_ids, $rule, $exclude_for ) {

			if ( 'vendor_list_excluded' === $exclude_for ) {
				if ( 'ywdpd_get_product_ids_involved_to_exclude' === current_filter() ) {
					$vendor_ids = $rule->get_meta( 'apply_adjustment_vendor_list_excluded' );
				} else {
					$vendor_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
				}
				$product_ids = $this->get_product_ids_by_vendor( $vendor_ids );
			}

			return $product_ids;
		}

		/**
		 * Return all product by vendor
		 *
		 * @param int[] $vendor_ids The vendor ids.
		 *
		 * @return int[]
		 */
		public function get_product_ids_by_vendor( $vendor_ids ) {
			if ( is_array( $vendor_ids ) && count( $vendor_ids ) > 0 ) {
				$tax_query = WC()->query->get_tax_query(
					array(
						array(
							'taxonomy' => $this->get_vendor_taxonomy_name(),
							'terms'    => array_values( $vendor_ids ),
							'operator' => 'IN',
						),
					)
				);

				return get_posts(
					array(
						'numberposts' => - 1,
						'post_type'   => array( 'product' ),
						'post_status' => 'publish',
						'tax_query'   => $tax_query, // phpcs:ignore WordPress.DB.SlowDBQuery
						'fields'      => 'ids',
					)
				);
			}
		}

		/**
		 * Add or remove the quantity for vendor products
		 *
		 * @param int              $qty       The comulative quantity.
		 * @param string           $rule_type The rule type.
		 * @param YWDPD_Price_Rule $rule      The rule.
		 *
		 * @return int
		 * @since  3.0.0
		 */
		public function check_comulative_quantity( $qty, $rule_type, $rule ) {
			$qt_ex = 0;
			if ( 'vendor_list' === $rule_type ) {
				$list_ids = $rule->get_meta( 'apply_to_vendors_list' );
				$qty      = YWDPD_Counter::get_comulative_taxonomy_count( $list_ids, $this->get_vendor_taxonomy_name() );
			}
			$is_exclude_activated = $rule->is_exclude_from_apply_enabled();

			if ( $is_exclude_activated ) {
				$ex_type = $rule->get_exclude_rule_for();

				if ( 'vendor_list_excluded' === $ex_type ) {
					$list_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
					$qt_ex    = YWDPD_Counter::get_comulative_taxonomy_count( $list_ids, $this->get_vendor_taxonomy_name() );

				}
			}

			return $qty - $qt_ex;
		}

		/**
		 * Add in the advanced conditions rules options in settings panels
		 *
		 * @param array $rules The cart rule options.
		 *
		 * @return array
		 */
		public function add_advanced_condition_option( $rules ) {

			if ( 'ywdpd_advanced_conditions_product_include_fields' === current_filter() ) {

				$options = array(
					array(
						'id'        => 'enable_require_product_vendors',
						'name'      => __( 'Require specific vendors in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to require products of specific vendors in the cart to apply the discount', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product require_product',
						'class'     => 'ywdpd_enable_require_product_vendors',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'require_product',
									),
								)
							),
						),
					),
					array(
						'id'        => 'require_product_vendors_list',
						'name'      => __( 'Include a list of vendors', 'ywdpd' ),
						'desc'      => __( 'Choose which vendors are required in cart to apply the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'             => $this->get_vendor_taxonomy_name(),
							'placeholder'          => __( 'Search for a vendor', 'ywdpd' ),
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'require_product',
									),
									array(
										'id'    => 'ywdpd_enable_require_product_vendors',
										'value' => 'yes',
									),
								)
							),
						),
						'class_row' => 'product require_product enable_require_product_vendors_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_advanced_conditions_product_exclude_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_exclude_product_vendors',
						'name'      => __( 'Exclude specific vendors from discount validation', 'ywdpd' ),
						'desc'      => __( 'Enable if you want to exclude specific vendors from this cart discount validation', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product exclude_product',
						'class'     => 'ywdpd_enable_exclude_product_vendors',
						'data'      => array(
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'exclude_product',
									),
								)
							),
						),
					),
					array(
						'id'        => 'exclude_product_vendors_list',
						'name'      => __( 'Exclude a list of vendors', 'ywdpd' ),
						'desc'      => __( 'Choose which product vendors to exclude from this cart validation', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'             => $this->get_vendor_taxonomy_name(),
							'placeholder'          => __( 'Search for a vendor', 'ywdpd' ),
							'ywdpd-condition-deps' => wp_json_encode(
								array(
									array(
										'id'    => 'ywdpd_condition_for',
										'value' => 'product',
									),
									array(
										'id'    => 'ywdpd_product_type',
										'value' => 'exclude_product',
									),
									array(
										'id'    => 'ywdpd_enable_exclude_product_vendors',
										'value' => 'yes',
									),
								)
							),
						),
						'class_row' => 'product exclude_product enable_exclude_product_vendors_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			}

			return $rules;
		}

		/**
		 * Valid the cart rule for vendors
		 *
		 * @param bool                             $is_valid   If rule is valid.
		 * @param array                            $condition  The specific condition.
		 * @param array                            $conditions All conditions.
		 * @param YWDPD_Cart_Rule|YWDPD_Last_Deals $rule       The cart rule.
		 *
		 * @return bool
		 */
		public function yith_vendor_valid_require_vendor_condition( $is_valid, $condition, $conditions, $rule ) {

			$type_check = $rule->get_condition_product_type( $condition );

			if ( ! $is_valid && 'require_product' === $type_check ) {
				$require_product_vendor = yith_plugin_fw_is_true( $condition['enable_require_product_vendors'] );
				$vendor_to_check        = $condition['require_product_vendors_list'] ?? array();
				if ( $require_product_vendor && is_array( $vendor_to_check ) && count( $vendor_to_check ) > 0 ) {

					$is_valid = YWDPD_Counter::check_taxonomy_in_cart( $vendor_to_check, $this->get_vendor_taxonomy_name(), 'at_least' );
				}
			}

			return $is_valid;
		}


		/**
		 * Check if the vendor is excluded
		 *
		 * @param bool            $is_excluded If the rule is excluded.
		 * @param array           $condition   The condition.
		 * @param WC_Product      $product     The product.
		 * @param YWDPD_Cart_Rule $rule        The rule.
		 *
		 * @return bool
		 */
		public function yith_vendor_valid_exclude_vendor_condition( $is_excluded, $condition, $product, $rule ) {

			$exclude_product_vendor = yith_plugin_fw_is_true( $condition['enable_exclude_product_vendors'] );
			$vendor_id_to_exclude   = $condition['exclude_product_vendors_list'] ?? array();

			if ( ! $is_excluded ) {
				if ( $exclude_product_vendor && count( $vendor_id_to_exclude ) > 0 ) {
					$product_terms = YWDPD_Taxonomy::get_product_term_ids( $product, $this->get_vendor_taxonomy_name() );
					$is_excluded   = count( array_intersect( $product_terms, $vendor_id_to_exclude ) ) > 0;
				}
			}

			return $is_excluded;
		}

		/**
		 * Add the product id by vendor
		 *
		 * @param int[]            $product_ids The ids list.
		 * @param YWDPD_Price_Rule $rule        The rule.
		 *
		 * @return int[]
		 */
		public function add_product_in_data_store( $product_ids, $rule ) {
			if ( 'vendor_list' === $rule->get_rule_for() ) {
				$vendor_ids  = $rule->get_meta( 'apply_to_vendors_list' ) ?? array();
				$list_ids    = $this->get_product_ids_by_vendor( $vendor_ids );
				$product_ids = array_merge( $product_ids, $list_ids );
			}

			return $product_ids;
		}

		/**
		 * Add the product id by vendor
		 *
		 * @param int[]            $excluded_product_ids The ids list.
		 * @param YWDPD_Price_Rule $rule                 The rule.
		 *
		 * @return int[]
		 */
		public function add_excluded_product_in_data_store( $excluded_product_ids, $rule ) {
			if ( $rule->is_exclude_from_apply_enabled() ) {
				if ( 'vendor_list' === $rule->get_exclude_rule_for() ) {
					$vendor_ids           = $rule->get_meta( 'apply_to_vendors_list_excluded' ) ?? array();
					$list_ids             = $this->get_product_ids_by_vendor( $vendor_ids );
					$excluded_product_ids = array_merge( $excluded_product_ids, $list_ids );
				}
			}

			return $excluded_product_ids;
		}

		/**
		 * Check if the product change the vendor
		 *
		 * @param bool       $has_changes The product has changes.
		 * @param WC_Product $product     The product.
		 *
		 * @return bool
		 */
		public function can_process_schedule_in_edit_product( $has_changes, $product ) {
			if ( ! $has_changes ) {
				$vendor = yith_wcmv_get_vendor( $product, 'product' );
				// phpcs:ignore WordPress.Security.NonceVerification.Missing
				$post_vendor_slug = isset( $_POST['tax_input']['yith_shop_vendor'] ) ? sanitize_text_field( wp_unslash( $_POST['tax_input']['yith_shop_vendor'] ) ) : '';
				$has_changes      = $vendor->get_slug() !== $post_vendor_slug;
			}

			return $has_changes;
		}
	}
}
