<?php
/**
 * Compatibility with YITH WooCommerce Product Add-ons
 *
 * @package YITH\DynamicPricingAndDiscount\Classes
 */

! defined( 'ABSPATH' ) && exit; // Exit if accessed directly.

/**
 * Class that manange the compatibility with Product Add-ons
 */
class YITH_YWDPD_Product_Add_Ons_Integration {
	/**
	 * Constructor
	 *
	 * Initialize class and registers actions and filters to be used
	 *
	 * @since  3.6.0
	 * @author YITH <plugins@yithemes.com>
	 */
	public function __construct() {
		add_filter( 'yith_wapo_blocks_product_price', array( $this, 'yith_wapo_fix_block_price_with_dynamic' ), 20, 3 );
		add_filter( 'yith_wapo_product_price', array( $this, 'yith_wapo_fix_block_price_with_dynamic' ), 20, 2 );
		add_filter( 'ywdpd_set_price_base', array( $this, 'yith_wapo_discount_only_main_price' ), 99, 2 );
		add_action( 'ywdpd_after_calculate_discounts', array( $this, 'yith_wapo_add_options_price' ), 99 );
		add_filter( 'ywdpd_cart_item_display_price', array( $this, 'yith_wapo_fix_display_price' ), 99, 2 );
		add_filter( 'ywdpd_cart_item_adjusted_price', array( $this, 'yith_wapo_fix_adjusted_price' ), 99, 2 );
	}


	/**
	 * Fix the addons price in product page
	 *
	 * @param float                     $price The price.
	 * @param WC_Product                $product The product.
	 * @param WC_Product_Variation|bool $variation The variation product.
	 *
	 * @return float|int|string
	 * @since 3.6.0
	 */
	public function yith_wapo_fix_block_price_with_dynamic( $price, $product, $variation = false ) {
		$pp        = $variation ? $variation : $product;
		$new_price = ywdpd_dynamic_pricing_discounts()->get_frontend_manager()->get_quantity_price( $price, $pp );
		if ( 'yith_wapo_product_price' === current_filter() ) {
			$new_price = wc_get_price_to_display( $product, array( 'price' => $new_price ) );
		}

		return $price !== $new_price ? $new_price : $price;
	}

	/**
	 * Discount only the main price
	 *
	 * @param float $price_to_discount The price.
	 * @param array $cart_item The cart item.
	 *
	 * @return float
	 * @since 3.6.0
	 */
	public function yith_wapo_discount_only_main_price( $price_to_discount, $cart_item ) {
		if ( isset( $cart_item['yith_wapo_item_price'] ) ) {
			$price_to_discount = $cart_item['yith_wapo_item_price'];
		}

		return $price_to_discount;
	}

	/**
	 * Fix the product prices with the addons
	 *
	 * @since 3.6.0
	 */
	public function yith_wapo_add_options_price() {
		if ( ! is_null( WC()->cart ) ) {
			foreach ( WC()->cart->get_cart_contents() as $cart_key => $cart_item ) {
				if ( isset( $cart_item['ywdpd_discounts'] ) && isset( $cart_item['yith_wapo_total_options_price'] ) ) {
					$discounted_price = $cart_item['ywdpd_discounts']['price_adjusted'];
					$cart_item['data']->set_price( $discounted_price + $cart_item['yith_wapo_total_options_price'] );
				}
			}
		}
	}

	/**
	 * Fix the right original price
	 *
	 * @param float $price The price.
	 * @param array $cart_item The cart item.
	 *
	 * @return float
	 * @since 3.6.0
	 */
	public function yith_wapo_fix_display_price( $price, $cart_item ) {

		if ( isset( $cart_item['yith_wapo_total_options_price'], $cart_item['yith_wapo_item_price'] ) ) {
			$price = $cart_item['yith_wapo_item_price'] + $cart_item['yith_wapo_total_options_price'];
		}

		return $price;
	}

	/**
	 * Fix the discounted  price
	 *
	 * @param float $adjusted_price The price.
	 * @param array $cart_item The cart item.
	 *
	 * @return float
	 * @since 3.6.0
	 */
	public function yith_wapo_fix_adjusted_price( $adjusted_price, $cart_item ) {
		if ( isset( $cart_item['yith_wapo_total_options_price'] ) ) {
			$adjusted_price = $adjusted_price + $cart_item['yith_wapo_total_options_price'];
		}

		return $adjusted_price;
	}
}
