<?php

defined( 'ABSPATH' ) || die();

Class Actions{

	const MODULES_NONCE = 'dnxte_nonce';

    public function __construct() {
		if ( ( defined( 'DOING_AJAX' ) && DOING_AJAX  )) {
			add_action( 'wp_ajax_get_popup_posts', array( $this, 'get_popup_posts' ) );
			add_action( 'wp_ajax_nopriv_get_popup_posts', array( $this, 'get_popup_posts' ) );
		}
    }

    /**
	 * Retrieves and filters posts based on site area and search criteria
	 *
	 * @since 1.0.0
	 * @return array|void Array of filtered posts or void if invalid request
	 */
	public function get_popup_posts() {
		$nonce = isset($_POST[self::MODULES_NONCE]) ? sanitize_text_field(wp_unslash($_POST[self::MODULES_NONCE])) : '';
		if (empty($_POST[self::MODULES_NONCE]) || !wp_verify_nonce($nonce, self::MODULES_NONCE)) {
			return;
		}

		// Retrieve and validate search query
		$search_query = isset($_POST['q']) ? sanitize_text_field(wp_unslash($_POST['q'])) : '';
		if (empty($search_query)) {
			return;
		}

		$nonce = isset($_POST['dnxte_nonce']) ? sanitize_text_field($_POST['dnxte_nonce']) : '';
		$validated_query = $this->get_validated_search_query($nonce);
		if (empty($validated_query)) {
			return;
		}

		// Determine response format and selected area
		$is_json_response = !empty($_POST['json']) ? (bool) sanitize_text_field($_POST['json']) : false;
		$selected_area = $this->get_validated_selected_area($nonce);

		// Retrieve filtered post types and query posts
		$post_types = $this->get_filtered_post_types($selected_area);
		$query_results = $this->query_posts($validated_query, $post_types);
		$processed_posts = $this->dnxte_convert_keys_to_lowercase($query_results['posts']);

		// Return response
		if ($is_json_response) {
			return $this->send_json_response($query_results['count'], $processed_posts);
		}

		return $processed_posts;
	}

	// Function to validate and retrieve the search query from the POST request
	private function get_validated_search_query($nonce) {
		// Check if 'q' is set in POST or if nonce is valid, then sanitize the search query
		$search_query = isset($_POST['q']) || wp_verify_nonce(sanitize_key($nonce), 'get_popup_posts')
			? sanitize_text_field(wp_unslash($_POST['q']))
			: '';

		// Return the sanitized search query, removing any slashes
		return $search_query ? stripslashes($search_query) : '';
	}

	// Function to validate and retrieve the selected area from the POST request
	private function get_validated_selected_area($nonce) {
		// Check if 'select_page' is set in POST or if nonce is valid, then sanitize the selected area
		return isset($_POST['select_page']) || wp_verify_nonce(sanitize_key($nonce), 'get_popup_posts')
			? sanitize_text_field(wp_unslash($_POST['select_page']))
			: '';
	}

	// Function to get filtered post types based on the selected area
	private function get_filtered_post_types($selected_area) {
		// Determine allowed post types based on whether the selected area is 'sitewide'
		$allowed_types = ($selected_area === 'sitewide')
			? get_post_types(array('public' => true))
			: array($selected_area);

		// Define post types to be excluded from the results
		$excluded_types = array(
			'attachment',
			'revision',
			'nav_menu_item',
			'custom_css',
			'et_pb_layout'
		);

		// Return the difference between allowed and excluded post types
		return array_diff($allowed_types, $excluded_types);
	}

	// Function to query posts based on search query and post types
	private function query_posts($search_query, $post_types) {
		// Set up query arguments for WP_Query
		$query_args = array(
			's' => $search_query,
			'post_type' => $post_types,
			'search_columns' => ['post_title'],
			'cache_results' => false,
			'posts_per_page' => 15,
			'orderby' => 'id',
			'order' => 'DESC'
		);

		// Execute the query
		$posts_query = new \WP_Query($query_args);

		// Return the posts and the total count of found posts
		return array(
			'posts' => $posts_query->get_posts(),
			'count' => (int) $posts_query->found_posts
		);
	}

	// Function to send a JSON response with the count and posts
	private function send_json_response($count, $posts) {
		// Set the content type to JSON
		header('Content-type: application/json');
		// Prepare the response data
		$response_data = array(
			'total_count' => $count,
			'items' => $posts
		);
		// Encode the response data as JSON and terminate the script
		die(wp_json_encode($response_data));
	}

	/**
     * Convert all keys in a multi-dimensional array or object to lowercase
     * Function Call Location: core/controllers/actions.php
     * @param array|object $data
     *
     * @return array|object
     */
    private function dnxte_convert_keys_to_lowercase($data) {
        if (!is_array($data) && !is_object($data)) {
            return $data;
        }

        $isObject = is_object($data);
        $convertedData = $isObject ? new stdClass() : [];

        foreach ($data as $key => $value) {
            $convertedValue = $this->dnxte_convert_keys_to_lowercase($value);
            $lowercaseKey = is_int($key) ? $key : strtolower($key);

            if ($isObject) {
                $convertedData->{$lowercaseKey} = $convertedValue;
            } else {
                $convertedData[$lowercaseKey] = $convertedValue;
            }
        }

        // Replace original data with the modified version
        if ($isObject) {
            foreach ($data as $key => $value) {
                unset($data->{$key});
            }
            foreach ($convertedData as $key => $value) {
                $data->{$key} = $value;
            }
        } else {
            $data = $convertedData;
        }

        return $data;
    }


}

if ( class_exists( 'Actions' ) ) {
	new Actions();
}
