<?php

defined('ABSPATH') || die();

/**
 * Class DiviShortcodeExtension
 */

class DiviShortcodeExtension {

    const SHORTCODE_TAG = 'divi_shortcode_library'; // Moved constant to class level

    public function __construct() {
        // Define shortcode if not already defined (for robustness)
        if (!defined('DE_SHORTCODE')) {
            define('DE_SHORTCODE', self::SHORTCODE_TAG); // Use class constant
        }

        // Register the shortcode handler
        add_shortcode(DE_SHORTCODE, array($this, 'render_divi_shortcode'));

        // Check if the extension is inactive
        $inactive_extensions = get_option('dnxte_inactive_extensions', array());
        if (!in_array('dnxte-shortcode-extension', $inactive_extensions)) {
            // Add columns to the Divi Layout post type table
            add_filter('manage_edit-et_pb_layout_columns', array($this, 'add_shortcode_column_header'));
            add_action('manage_et_pb_layout_posts_custom_column', array($this, 'populate_shortcode_column_content'));
        }
    }

    /**
     * Renders the Divi Shortcode.
     *
     * @param array $attributes Shortcode attributes, including 'id'.
     * @return string The shortcode output.
     */
    public function render_divi_shortcode($attributes = array()) {
        if (empty($attributes['id'])) {
            return ''; // Return empty string if no ID is provided. Prevents errors.
        }

        // Sanitize the ID.  Important to prevent potential exploits.
        $id = intval($attributes['id']);

        // Generate the Divi module shortcode. Using sprintf for better readability
        $shortcode = sprintf('[et_pb_section global_module="%d"][/et_pb_section]', $id);

        // Return the executed shortcode
        return do_shortcode($shortcode);
    }

    /**
     * Adds a "Shortcode Library" column to the Divi Layout post type list table.
     *
     * @param array $columns An array of existing columns.
     * @return array An array of columns with the new shortcode column added.
     */
    public function add_shortcode_column_header($columns) {
        $columns['shortcode'] = __('Shortcode ID', 'dnxte-divi-essential');
        return $columns;
    }

    /**
     * Populates the "Shortcode Library" column with the shortcode for the Divi Layout.
     *
     * @param string $column_name The name of the column being populated.
     */
    public function populate_shortcode_column_content($column_name) {
        global $post;

        if ($column_name === 'shortcode') {
            // Generate the shortcode
            $shortcode = esc_attr(sprintf('[%s id="%d"]', DE_SHORTCODE, $post->ID));

            // Output the shortcode input field with inline styles moved to CSS
            printf('<input class="dnxte-shortcode-input" type="text" readonly onfocus="this.select()" value="%s" style="width:100%%" />', esc_attr($shortcode));
        }
    }
}

// Instantiate the class.  Important for the code to actually work.
$divi_shortcode_extension = new DiviShortcodeExtension();