<?php

defined('ABSPATH') || exit;

/**
 * Class Dnxte_Library_Widget
 *
 * A WordPress widget to display a saved Divi Library layout.
 *
 * @package Divi Essential
 * @since 1.0.0
 */
class dnxte_widget_library extends WP_Widget
{

	/**
	 * Constructor
	 */
	public function __construct()
	{
		parent::__construct(
			'dnxte_library_widget',
			__('Divi Library Widget', 'dnxte-divi-essential'),
			array(
				'description' => __('Your saved Divi Library layouts show in the widget.', 'dnxte-divi-essential'),
			)
		);
	}

	/**
	 * Widget front-end display.
	 *
	 * @param array $args     Widget arguments.
	 * @param array $instance Widget instance settings.
	 */
	public function widget($args, $instance)
	{
		// Check if Divi Builder is available.
		if (! class_exists('ET_Builder_Element')) {
			return;
		}

		$widget_title = ! empty($instance['title']) ? apply_filters('widget_title', $instance['title']) : '';
		$layout_name  = ! empty($instance['library_name']);
		$layout_id    = ! empty($instance['library']) ? $instance['library'] : '';

		echo et_core_intentionally_unescaped($args['before_widget'], 'html');

		// Display title based on settings.
		if ($layout_name && ! empty($layout_id)) {
			$widget_title = get_the_title($layout_id);
		}

		if (! empty($widget_title)) {
			echo et_core_intentionally_unescaped($args['before_title'] . esc_html($widget_title) . $args['after_title'], 'html');
		}

		// Render Divi Library layout.
		if (! empty($layout_id) && 'none' !== $layout_id) {
			echo do_shortcode('[divi_shortcode_library id="' . esc_attr($layout_id) . '"]');
		}

		echo et_core_intentionally_unescaped($args['after_widget'], 'html');
	}

	/**
	 * Widget backend form.
	 *
	 * @param array $instance Current widget settings.
	 */
	public function form($instance)
	{
		$widget_title = isset($instance['title']) ? esc_attr($instance['title']) : '';
		$layout_id    = isset($instance['library']) ? esc_attr($instance['library']) : '';
		$layout_name  = ! empty($instance['library_name']);
		$divi_library = $this->get_divi_library_options();
		?>

		<p>
			<label for="<?php echo esc_attr($this->get_field_id('title')); ?>"><?php esc_html_e('Title:', 'dnxte-divi-essential'); ?></label>
			<input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>"
				name="<?php echo esc_attr($this->get_field_name('title')); ?>"
				type="text" value="<?php echo esc_attr($widget_title); ?>" />
		</p>
		<p>
			<input class="checkbox" type="checkbox" <?php checked($layout_name, true); ?>
				id="<?php echo esc_attr($this->get_field_id('library_name')); ?>"
				name="<?php echo esc_attr($this->get_field_name('library_name')); ?>" />
			<label for="<?php echo esc_attr($this->get_field_id('library_name')); ?>">
				<?php esc_html_e('Use Divi Library Layout Name as Title', 'dnxte-divi-essential'); ?>
			</label>
		</p>
		<p>
			<label for="<?php echo esc_attr($this->get_field_id('library')); ?>"><?php esc_html_e('Select Divi Library Layout:', 'dnxte-divi-essential'); ?></label>
			<select class="widefat" id="<?php echo esc_attr($this->get_field_id('library')); ?>"
				name="<?php echo esc_attr($this->get_field_name('library')); ?>">
				<?php foreach ($divi_library as $label => $id) : ?>
					<option value="<?php echo esc_attr($id); ?>" <?php selected($id, $layout_id); ?>>
						<?php echo esc_html($label); ?>
					</option>
				<?php endforeach; ?>
			</select>
		</p>
	<?php
	}

	/**
	 * Save widget settings.
	 *
	 * @param array $new_instance New widget settings.
	 * @param array $old_instance Old widget settings.
	 *
	 * @return array Updated settings.
	 */
	public function update($new_instance, $old_instance)
	{
		$instance                 = array();
		$instance['title']        = sanitize_text_field($new_instance['title']);
		$instance['library_name'] = ! empty($new_instance['library_name']) ? true : false;
		$instance['library']      = sanitize_text_field($new_instance['library']);

		return $instance;
	}

	/**
	 * Get Divi Library options.
	 *
	 * @return array List of available Divi Library layouts.
	 */
	private function get_divi_library_options()
	{
		$library_options = array(__('Select Library Layout', 'dnxte-divi-essential') => 'none');

		$args = array(
			'post_type'      => 'et_pb_layout',
			'posts_per_page' => -1,
			'orderby'        => 'title',
			'order'          => 'ASC',
			'fields'         => array('ID', 'post_title'),
		);

		$layouts = get_posts($args);

		if (! empty($layouts)) {
			foreach ($layouts as $layout) {
				$library_options[$layout->post_title] = $layout->ID;
			}
		}

		return $library_options;
	}
}