<?php
/**
 * Handing licese features
 *
 * @link       Artbees.net
 * @since      1.0.0
 *
 * @package    Sellkit
 * @subpackage Sellkit/admin
 */

defined( 'ABSPATH' ) || exit;

/**
 * Sellkit License.
 *
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 * @SuppressWarnings(PHPMD.NPathComplexity)
 */
class Sellkit_License {

	/**
	 * Hook constants.
	 */
	const SYNC_HOOK = 'sellkit_site_sync';

	/**
	 * Sellkit Website URL
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sellkit_site;

	/**
	 * Admin url.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $admin_url;

	/**
	 * Site url.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $site_url;

	/**
	 * Product ID that will be activated.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $product_id;

	/**
	 * Stores license data.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sellkit_license_option;

	/**
	 * Sellkit license admin page.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public $sellkit_license_page;

	/**
	 * Class construct.
	 */
	public function __construct() {

		$this->sellkit_site           = 'https://my.getsellkit.com/';
		$this->product_id             = 'prod_KRQzQ84ZyrIzXy';
		$this->sellkit_license_option = 'sellkit_license';
		$this->sellkit_license_page   = 'sellkit-license';
		$this->admin_url              = is_multisite() ? network_admin_url( 'admin.php?page=' . $this->sellkit_license_page ) : admin_url( 'admin.php?page=' . $this->sellkit_license_page );
		$this->site_url               = home_url();

	}

	/**
	 * Bind license-related hooks.
	 *
	 * @return void
	 */
	public static function bind() {
		add_action( self::SYNC_HOOK, [ self::class, 'sync_site' ] );
		add_action( 'admin_notices', [ self::class, 'render_license_notices' ] );
		add_action( 'network_admin_notices', [ self::class, 'render_license_notices' ] );
		add_action( 'admin_init', [ self::class, 'cleanup_expired_license' ] );
		// New site creation hooks.
		add_action( 'wp_insert_site', [ self::class, 'schedule_for_new_site' ], 10, 1 );
		add_action( 'wpmu_new_blog', [ self::class, 'schedule_for_new_site' ], 10, 1 );
	}

	/**
	 * Schedule sync for a specific site or current site.
	 *
	 * @param int|null $site_id Site ID or null for current site.
	 * @return void
	 */
	public static function schedule_for_site( $site_id = null ) {
		$switched = false;
		if ( is_multisite() && null !== $site_id && get_current_blog_id() !== (int) $site_id ) {
			switch_to_blog( (int) $site_id );
			$switched = true;
		}
		if ( ! wp_next_scheduled( self::SYNC_HOOK ) ) {
			wp_schedule_event( time(), 'weekly', self::SYNC_HOOK );
		}
		if ( $switched ) {
			restore_current_blog();
		}
	}

	/**
	 * Clear sync for a specific site or current site.
	 *
	 * @param int|null $site_id Site ID or null for current site.
	 * @return void
	 */
	public static function clear_for_site( $site_id = null ) {
		$switched = false;
		if ( is_multisite() && null !== $site_id && get_current_blog_id() !== (int) $site_id ) {
			switch_to_blog( (int) $site_id );
			$switched = true;
		}
		wp_clear_scheduled_hook( self::SYNC_HOOK );
		if ( $switched ) {
			restore_current_blog();
		}
	}

	/**
	 * Schedule sync on the main site (network-level scheduling).
	 *
	 * Falls back to current site on single-site installs.
	 *
	 * @return void
	 */
	public static function schedule_network() {
		// Prefer scheduling across all sites on multisite.
		if ( is_multisite() ) {
			self::schedule_for_all_sites();
			return;
		}
		self::schedule_for_site();
	}

	/**
	 * Clear sync on the main site (network-level clearing).
	 *
	 * Falls back to current site on single-site installs.
	 *
	 * @return void
	 */
	public static function clear_network() {
		// Prefer clearing across all sites on multisite.
		if ( is_multisite() ) {
			self::clear_for_all_sites();
			return;
		}
		self::clear_for_site();
	}

	/**
	 * Schedule sync on all sites in the network.
	 *
	 * @return void
	 */
	public static function schedule_for_all_sites() {
		if ( ! is_multisite() ) {
			self::schedule_for_site();
			return;
		}
		$site_ids = get_sites( [
			'number' => 0,
			'fields' => 'ids',
		] );
		if ( empty( $site_ids ) || ! is_array( $site_ids ) ) {
			return;
		}
		foreach ( $site_ids as $site_id ) {
			self::schedule_for_site( (int) $site_id );
		}
	}

	/**
	 * Clear sync on all sites in the network.
	 *
	 * @return void
	 */
	public static function clear_for_all_sites() {
		if ( ! is_multisite() ) {
			self::clear_for_site();
			return;
		}
		$site_ids = get_sites( [
			'number' => 0,
			'fields' => 'ids',
		] );
		if ( empty( $site_ids ) || ! is_array( $site_ids ) ) {
			return;
		}
		foreach ( $site_ids as $site_id ) {
			self::clear_for_site( (int) $site_id );
		}
	}

	/**
	 * Handle new site creation (modern hook).
	 *
	 * @param int|WP_Site $site Site object or site ID.
	 * @return void
	 */
	public static function schedule_for_new_site( $site ) {
		$site_id = is_object( $site ) && isset( $site->blog_id ) ? (int) $site->blog_id : (int) $site;
		if ( $site_id > 0 ) {
			self::schedule_for_site( $site_id );
		}
	}

	/**
	 * Get option.
	 *
	 * @param string $name name.
	 * @since 1.2.0
	 */
	public function get( $name ) {
		$option = get_site_option( $this->sellkit_license_option );

		if ( empty( $option[ $name ] ) ) {
			return false;
		}

		return $option[ $name ];
	}

	/**
	 * Get Secure URL to remote access to register license.
	 *
	 * @since 1.0.0
	 */
	public function switch() {

		if ( is_multisite() ) {
			$main_site_id = function_exists( 'get_main_site_id' ) ? get_main_site_id() : 1;
			if ( get_current_blog_id() !== (int) $main_site_id ) {
				$redirect_url = network_admin_url( 'admin.php?page=' . $this->sellkit_license_page );
				wp_redirect( $redirect_url ); //phpcs:ignore
				exit;
			}
		}

		$result = wp_remote_post( $this->sellkit_site . '/wp-json/sellkit/v1/license/delete', [
			'body' => [
				'site_key'     => $this->get( 'site_key' ),
				'secret_token' => $this->get( 'secret_token' ),
			],
			'headers' => [
				'-sellkit-client-id' => $this->get( 'client_id' ),
				'-sellkit-auth-token' => $this->get( 'auth_token' ),
			],
		] );

		$body = json_decode( wp_remote_retrieve_body( $result ), true );

		if ( isset( $body['status'] ) ) {
			delete_site_option( $this->sellkit_license_option );
			delete_site_option( 'sellkit_dismiss_license_notice' );
		}

		$this->activate();
	}

	/**
	 * Get Secure URL to remote access to register license.
	 *
	 * @since 1.0.0
	 */
	public function activate() {

		$current_theme = wp_get_theme();
		$nonce         = wp_create_nonce( $this->sellkit_license_page );
		$session_token = $this->create_session_token();
		$partner_id    = $this->get_partner_id();

		$url_params = [
			'action'        => 'connect',
			'site_url'      => urlencode( $this->site_url ), //phpcs:ignore
			'redirect_url'  => urlencode( $this->admin_url ), //phpcs:ignore
			'product_id'    => $this->product_id,
			'theme'         => $current_theme->get( 'Name' ),
			'version'       => SELLKIT_PRO_VERSION, // current Sellkit Plugin Version.
			'session_token' => $session_token,
			'nonce'         => $nonce,
		];

		if ( $partner_id ) {
			$url_params['partner_id'] = $partner_id;
		}

		$url = add_query_arg( $url_params, $this->sellkit_site . '/authorize' );

		wp_redirect( $url ); //phpcs:ignore
	}

	/**
	 * Deactivate license from client and Sellkit server.
	 *
	 * @since 1.0.0
	 */
	public function deactivate() {

		if ( is_multisite() ) {
			$main_site_id = function_exists( 'get_main_site_id' ) ? get_main_site_id() : 1;
			if ( get_current_blog_id() !== (int) $main_site_id ) {
				$redirect_url = network_admin_url( 'admin.php?page=' . $this->sellkit_license_page );
				wp_redirect( $redirect_url ); //phpcs:ignore
				exit;
			}
		}

		$result = wp_remote_post( $this->sellkit_site . '/wp-json/sellkit/v1/license/delete', [
			'body' => [
				'site_key'     => $this->get( 'site_key' ),
				'secret_token' => $this->get( 'secret_token' ),
			],
			'headers' => [
				'-sellkit-client-id' => $this->get( 'client_id' ),
				'-sellkit-auth-token' => $this->get( 'auth_token' ),
			],
		] );

		$body = json_decode( wp_remote_retrieve_body( $result ), true );

		delete_site_option( $this->sellkit_license_option );
		delete_site_option( 'sellkit_dismiss_license_notice' );

		wp_redirect( $this->admin_url ); //phpcs:ignore
		exit;
	}

	/**
	 * Get the secure URL that will route the app to the appropriate action.
	 *
	 * @param string $action action name.
	 * @since 1.0.0
	 */
	public function get_url( $action ) {

		$url = add_query_arg([
			'page'   => $this->sellkit_license_page,
			'action' => $action,
			'nonce'  => wp_create_nonce( $this->sellkit_license_page ),
		], is_multisite() ? network_admin_url( 'admin.php' ) : admin_url( 'admin.php' ) );

		return $url;
	}

	/**
	 * Connect after successful authorization.
	 *
	 * @since 1.0.0
	 */
	public function connect() {
		$user_email    = sellkit_htmlspecialchars( INPUT_GET, 'user_email' );
		$site_key      = sellkit_htmlspecialchars( INPUT_GET, 'site_key' );
		$secret_token  = sellkit_htmlspecialchars( INPUT_GET, 'secret_token' );
		$session_token = sellkit_htmlspecialchars( INPUT_GET, 'session_token' );
		$client_id     = sellkit_htmlspecialchars( INPUT_GET, 'client_id' );
		$auth_token    = sellkit_htmlspecialchars( INPUT_GET, 'auth_token' );

		if ( $this->verify_session_token( $session_token ) ) {

			update_site_option( $this->sellkit_license_option, [
				'status'       => 'active',
				'user_email'   => $user_email,
				'client_id'    => $client_id,
				'auth_token'   => $auth_token,
				'site_key'     => $site_key,
				'secret_token' => $secret_token,
			]);

			$this->destroy_session_token();

		}

		wp_redirect( $this->admin_url ); //phpcs:ignore

	}

	/**
	 * Create session token for imporved security.
	 *
	 * @return  string
	 * @since  1.0.0
	 */
	public function create_session_token() {

		$transient_name = 'sellkit_secure_token';
		$transient      = get_transient( $transient_name );

		if ( $transient ) {
			return $transient;
		}

		$token = $this->generate_token();

		set_transient( $transient_name, $token, 60 * 60 );

		return $token;

	}

	/**
	 * Verify session token.
	 *
	 * @param string $token token.
	 * @return boolean
	 * @since  1.0.0
	 */
	public function verify_session_token( $token ) {

		$transient_name = 'sellkit_secure_token';
		$transient      = get_transient( $transient_name );

		if ( ! $transient ) {
			return false;
		}

		if ( $token === $transient ) {
			return true;
		}
		return false;

	}

	/**
	 * Destroy session token.
	 *
	 * @return void
	 * @since  1.0.0
	 */
	public function destroy_session_token() {

		delete_transient( 'sellkit_secure_token' );

	}

	/**
	 * Generate a secret token.
	 *
	 * @return  string
	 * @since  1.0.0
	 */
	public function generate_token() {
		return hash_hmac( 'sha256', get_current_user_id(), 'ucEJgMKmo6NbZztjs' . time() );
	}

	/**
	 * Sync license/site status with remote service.
	 *
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 * @return void
	 */
	public static function sync_site() {
		$license = new self();

		if ( 'active' !== $license->get( 'status' ) ) {
			return;
		}

		$site_key     = $license->get( 'site_key' );
		$secret_token = $license->get( 'secret_token' );
		$client_id    = $license->get( 'client_id' );
		$auth_token   = $license->get( 'auth_token' );

		if ( false === $site_key || false === $secret_token || false === $client_id || false === $auth_token ) {
			return;
		}

		$response = wp_remote_post( $license->sellkit_site . '/wp-json/sellkit/v1/site/sync', [
			'headers' => [
				'-sellkit-client-id'  => $client_id,
				'-sellkit-auth-token' => $auth_token,
			],
			'body'    => [
				'site_key'        => $site_key,
				'secret_token'    => $secret_token,
				'current_version' => defined( 'SELLKIT_PRO_VERSION' ) ? SELLKIT_PRO_VERSION : null,
			],
		] );

		if ( is_wp_error( $response ) ) {
			return;
		}

		$body = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( empty( $body ) || ! is_array( $body ) ) {
			return;
		}

		$license_option = get_site_option( $license->sellkit_license_option );
		if ( ! is_array( $license_option ) ) {
			$license_option = [];
		}
		if ( isset( $body['site_status'] ) ) {
			if ( true === $body['site_status'] ) {
				$license_option['status'] = 'active';
			} else {
				$license_option['status'] = 'expired';
				unset( $license_option['site_key'], $license_option['secret_token'], $license_option['client_id'], $license_option['auth_token'], $license_option['user_email'] );
				// Ensure expired notice is shown again network-wide.
				delete_site_option( 'sellkit_dismiss_license_notice' );
			}
			update_site_option( $license->sellkit_license_option, $license_option );
		}
	}

	/**
	 * Cleanup stored credentials if license is expired.
	 *
	 * @return void
	 */
	public static function cleanup_expired_license() {
		$license = new self();

		if ( 'expired' !== $license->get( 'status' ) ) {
			return;
		}

		$license_option = get_site_option( $license->sellkit_license_option );
		if ( ! is_array( $license_option ) ) {
			return;
		}

		$changed = false;
		$keys    = [ 'site_key', 'secret_token', 'client_id', 'auth_token', 'user_email' ];
		foreach ( $keys as $key ) {
			if ( isset( $license_option[ $key ] ) ) {
				unset( $license_option[ $key ] );
				$changed = true;
			}
		}

		if ( $changed ) {
			update_site_option( $license->sellkit_license_option, $license_option );
			delete_site_option( 'sellkit_dismiss_license_notice' );
		}
	}

	/**
	 * Proxy to instance notice renderer for hooking as static.
	 *
	 * @return void
	 */
	public static function render_license_notices() {
		$license = new self();
		$license->license_problem_notices();
	}

	/**
	 * Show connect notice if not connected.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return void
	 */
	public function license_problem_notices() {

		$has_capability = is_multisite() ? current_user_can( 'manage_network_options' ) : current_user_can( 'manage_options' );
		if ( 'active' === $this->get( 'status' ) || ! $has_capability ) {
			return;
		}

		if ( false === $this->get( 'status' ) ) {
			if ( false !== get_site_option( 'sellkit_dismiss_license_notice' ) ) {
				return;
			}
			$message = sprintf(
				'<span style="display:block;margin:0;">'
				. '<h3>'
				. __( 'Welcome to Sellkit Pro', 'sellkit-pro' )
				. '</h3>'
				. __( 'Please activate your license to get plugin updates, premium support and access to some locked features.', 'sellkit-pro' )
				. '</span>'
			);

			$message .= sprintf(
				'<span style="display:block;margin-top:1em; clear: both;">' .
				'<a class="button-primary" href="%1$s">%2$s</a></span>',
				$this->get_url( 'activate' ),
				__( 'Connect & Activate', 'sellkit-pro' )
			);

			$is_dismissable_class = 'is-dismissible';
		}

		if ( 'expired' === $this->get( 'status' ) ) {

			if ( false !== get_site_option( 'sellkit_dismiss_license_notice' ) ) {
				return;
			}
			$message = sprintf(
				'<span style="display:block;margin:0;">'
				. '<h3>'
				. __( 'Your License has expired!', 'sellkit-pro' )
				. '</h3>'
				. __( 'Please renew your license to continue getting plugin updates, premium support and access to some locked features.', 'sellkit-pro' )
				. '</span>'
			);

			$message .= sprintf(
				'<span style="display:block;margin-top:1em; clear: both;">' .
				'<a class="button-primary" href="https://my.getsellkit.com/subscriptions">%s</a></span>',
				__( 'Renew License', 'sellkit-pro' )
			);

			$is_dismissable_class = '';

		}

		if ( 'invalid' === $this->get( 'status' ) ) {

			if ( false !== get_site_option( 'sellkit_dismiss_license_notice' ) ) {
				return;
			}
			$message = sprintf(
				'<span style="display:block;margin:0;">'
				. '<h3>'
				. __( 'Your License seems to be invalid!!', 'sellkit-pro' )
				. '</h3>'
				. __( 'Your license details(Site Key, Secret Token, Account Email) does not match our records. Please deactivate the license and then reactivate it again to resolve this problem.', 'sellkit-pro' )
				. '</span>'
			);

			$message .= sprintf(
				'<span style="display:block;margin-top:1em; clear: both;">' .
				'<a class="button-primary" href="%1$s">%2$s</a></span>',
				$this->get_url( 'activate' ),
				__( 'Deactivate & Activate', 'sellkit-pro' )
			);

			$is_dismissable_class = '';

		}

		if ( ! isset( $message ) ) {
			return;
		}

		printf(
			'<div data-nonce="%s" class="sellkit-notice notice sellkit-license-notice %s">%s<div class="sellkit-notice-content">%s</div></div>',
			wp_create_nonce( 'sellkit_dismiss_license_notice' ), // phpcs:ignore
			esc_attr( $is_dismissable_class ),
			'<div class="sellkit-notice-aside"><span class="sellkit-notice-aside-icon"><span></div>',
			$message //phpcs:ignore
		);
	}

	/**
	 * Set option if notice is dismissed.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return void
	 */
	public function dismiss_license_notice() {
		check_ajax_referer( 'sellkit_dismiss_license_notice' );
		update_site_option( 'sellkit_dismiss_license_notice', 1 );
	}

	/**
	 * Checks if plugin bundled and if so return partner id
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string
	 */
	public function get_partner_id() {

		if ( defined( 'SELLKIT_BUNDLED' ) ) {
			return SELLKIT_PARTNER_ID;
		}
		return; //phpcs:ignore
	}

	/**
	 * Checks if there site is connected to remote and the license status is active.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return boolean
	 */
	public function is_license_active() {

		if ( 'active' === $this->get( 'status' ) ) {
			return true;
		}

		return false;

	}

}

new Sellkit_License();
Sellkit_License::bind();
