<?php
/**
 * This file belongs to the YIT Plugin Framework.
 *
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.txt
 *
 * @package YITH Product Description in Loop for WooCommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_PDIL_Frontend' ) ) {

	/**
	 * Class YITH_PDIL_Frontend
	 */
	class YITH_PDIL_Frontend {

		/**
		 * Single instance of the class
		 *
		 * @var \YITH_PDIL_Frontend
		 * @since 1.0.0
		 */
		protected static $instance;

		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_PDIL_Frontend
		 * @since 1.0.0
		 */
		public static function get_instance() {
			return ! is_null( self::$instance ) ? self::$instance : self::$instance = new self();
		}

		/**
		 * Constructor
		 *
		 * @return  void
		 * @since   1.0.0
		 * @author Francesco Grasso - francgrasso@yithemes.com
		 */
		private function __construct() {

			$hook     = '';
			$priority = '';
			switch ( YITH_PDIL_Settings()->yith_pdil_get_option( 'description_position', 'after_title' ) ) {

				case 'before_title':
					$hook     = 'woocommerce_shop_loop_item_title';
					$priority = 5;
					break;
				case 'after_title':
					$hook     = 'woocommerce_shop_loop_item_title';
					$priority = 15;
					break;
				case 'before_price':
					$hook     = 'woocommerce_after_shop_loop_item_title';
					$priority = 8;
					break;
				case 'after_price':
					$hook     = 'woocommerce_after_shop_loop_item_title';
					$priority = 15;
					break;
				case 'before_rating':
					$hook     = 'woocommerce_after_shop_loop_item_title';
					$priority = 2;
					break;
				case 'after_rating':
					$hook     = 'woocommerce_after_shop_loop_item_title';
					$priority = 7;
					break;
				case 'before_add_to_cart':
					$hook     = 'woocommerce_after_shop_loop_item';
					$priority = 7;
					break;
				case 'after_add_to_cart':
					$hook     = 'woocommerce_after_shop_loop_item';
					$priority = 15;
					break;
				case 'custom':
					$hook     = YITH_PDIL_Settings()->yith_pdil_get_option( 'custom_hook', '' );
					$priority = YITH_PDIL_Settings()->yith_pdil_get_option( 'custom_priority', 10 );
					break;
			}

			add_action( $hook, array( $this, 'show_product_loop_description' ), $priority );

			if ( YITH_PDIL_Settings()->yith_pdil_get_option( 'use_plugin_style', 'no' ) == 'yes' ) {
				add_action( 'wp_head', array( $this, 'enqueue_scripts' ), 100 );
			}
		}

		/**
		 * Show or not the product description
		 *
		 * @author Francesco Grasso - francgrasso@yithemes.com
		 */
		public function show_product_loop_description() {
			global $post;
			$show_everywhere  = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_all_loops', 'no' );
			$show_in_shop     = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_shop_page', 'no' );
			$show_in_category = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_category_page', 'no' );
			$show_in_tag      = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_tag_page', 'no' );

			if ( 'yes' == $show_everywhere ) {
				YITH_PDIL_Frontend()->render_pdil( $post );
			} elseif ( is_shop() && 'yes' == $show_in_shop ) {
				YITH_PDIL_Frontend()->render_pdil( $post );
			} elseif ( is_product_category() && 'yes' == $show_in_category ) {
				YITH_PDIL_Frontend()->render_pdil( $post );
			} elseif ( is_product_tag() && 'yes' == $show_in_tag ) {
				YITH_PDIL_Frontend()->render_pdil( $post );
			}
		}

		/**
		 * Render the description
		 *
		 * @param object $post Product post object.
		 *
		 * @author Francesco Grasso - francgrasso@yithemes.com
		 */
		public function render_pdil( $post ) {
			$needs_trim   = ! YITH_PDIL_Settings()->yith_pdil_get_option( 'advanced_editor', 'yes' );
			$limit_words  = YITH_PDIL_Settings()->yith_pdil_get_option( 'limit_words', 55 );
			$allowed_tags = apply_filters( 'yith_pdil_allowed_tags', "'<ul><ol><li><dl><dt><b><strong><span><div><p><em><i><blockquote><quote><q><cite><del><stroke><ins>'" );

			$text = get_post_meta( $post->ID, '_yith_pdil_description', true );
			$text = strip_tags( html_entity_decode( $text ), $allowed_tags );
			if ( $needs_trim && $limit_words > 0 ) {
				$text = wp_trim_words( $text, $limit_words, $more = null );
			}
			if ( $text ) {
				echo do_shortcode( '<div class="yith_pdil">' . $text . '</div>' );
			}
		}

		/**
		 * Enqueue plugin scripts
		 */
		public function enqueue_scripts() {
			$show_in_shop     = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_shop_page', 'no' );
			$show_in_category = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_category_page', 'no' );
			$show_in_tag      = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_tag_page', 'no' );
			$show_everywhere  = YITH_PDIL_Settings()->yith_pdil_get_option( 'description_in_all_loops', 'no' );

			// Retrieve style options.
			$text_align   = YITH_PDIL_Settings()->yith_pdil_get_option( 'text_alignment', 'left' );
			$color        = YITH_PDIL_Settings()->yith_pdil_get_option( 'color', '#ebebeb' );
			$font_size    = YITH_PDIL_Settings()->yith_pdil_get_option( 'font_size', '16' );
			$custom_style = '<style>.yith_pdil {
					text-align: ' . $text_align . ';
					font-size: ' . $font_size . 'px;
					color: ' . $color . ';
				}</style>';

			if ( is_shop() && 'yes' == $show_in_shop ) {
				echo $custom_style;
			} elseif ( is_product_category() && 'yes' == $show_in_category ) {
				echo $custom_style;
			} elseif ( is_product_tag() && 'yes' == $show_in_tag ) {
				echo $custom_style;
			} elseif ( 'yes' == $show_everywhere ) {
				echo $custom_style;
			}

		}
	}
}
/**
 * Unique access to instance of YITH_PDIL_Frontend class
 *
 * @return \YITH_PDIL_Frontend
 * @since 1.0.0
 */
function YITH_PDIL_Frontend() {
	return YITH_PDIL_Frontend::get_instance();
}
