<?php
namespace Merkulove\Helper;

use Merkulove\Helper\Unity\Plugin;
use Merkulove\Helper\Unity\Settings;
use Orhanerday\OpenAi\OpenAi;
use Parsedown;
use Smalot\PdfParser\Parser;


/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

require Plugin::get_path() . '/vendor/autoload.php';

/**
* SINGLETON: AjaxActions class contains ajax logic.
*
* @since 1.0.0
*
**/
final class OpenAiBot {

    /**
     * The one true AjaxActions.
     *
     * @since 1.0.0
     * @access private
     * @var OpenAiBot
     **/
    private static $instance;



    /**
     * Creates prompt based on post content.
     *
     * @return string
     * @since 1.0.0
     * @access private
     *
     */
    private function get_post_content() {
        $options = Settings::get_instance()->options;
        if ( empty( $options['open_ai_post_id'] ) ) { return ''; }

        $posts_content = '';

        $posts_ids = $options['open_ai_post_id'];

        foreach ( $posts_ids as $posts_id ) {
            $post_content = get_post( esc_attr( str_replace( 'post_', '', $posts_id ) ) )->post_content;
            $post_content = apply_filters( 'the_content', $post_content );
            $post_content = str_replace(']]>', ']]&gt;', $post_content );
            $posts_content .= strip_tags( $post_content );
            $posts_content .= ' ';
        }

        return $posts_content;

    }

    /**
     * Returns product categories
     *
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function get_product_categories( $product_id ) {
        $product_cats = get_the_terms( $product_id, 'product_cat' );
        $categories = [];

        foreach ( $product_cats as $product_cat ) {
            $categories[] = $product_cat->name;
        }

        return implode( ', ', $categories );
    }

    /**
     * Returns prompt based on WooCommerce products
     *
     * @return string
     * @since 1.0.0
     * @access private
     */
    private function get_products_prompt() {
        if (  !in_array('woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
            return '';
        }

        $products = wc_get_products( [ 'status' => 'publish', 'limit' => -1  ] );
        $product_attrs = Settings::get_instance()->options['open_ai_product_attrs'];


        $products_prompt = 'Products: ';

        if ( !empty( $products ) ) {
            foreach ( $products as $product ) {
                $product_data = $product->get_data();
                $product_cats = $this->get_product_categories( $product_data['id'] );
                $in_stock = $product_data['stock_status'] === 'instock' ? 'yes' : 'no';
                $on_sale = !empty( $product_data['sale_price'] ) ? 'yes' : 'no';
                $products_prompt .= sprintf(
                    '%s %s %s %s %s %s %s',
                    in_array( 'name', $product_attrs ) ? 'Name: ' . $product_data['name'] . ',' : '',
                            in_array( 'description', $product_attrs ) ? 'Description: ' . $product_data['description'] . ',' : '',
                            in_array( 'category', $product_attrs ) ? 'Category: ' . $product_cats . ',' : '',
                            in_array( 'price', $product_attrs ) ?
                              'Price: ' .  $product_data['price'] . get_woocommerce_currency_symbol() . ',' :
                              '',
                            in_array( 'in_stock', $product_attrs ) ? 'In stock: ' . $in_stock . ',' : '',
                            in_array( 'on_sale', $product_attrs ) ? 'On sale: ' . $on_sale . ',' : '',
                            in_array( 'link', $product_attrs ) ? 'Link: ' . get_permalink( $product_data['id'] ) : ''
                );

                foreach ( $product->get_attributes() as $product_attribute ) {
                    if ( in_array( 'id_' . $product_attribute['id'], $product_attrs ) ) {
                        $products_prompt .=
                            ' ' . wc_get_attribute( $product_attribute['id'] )->name . ': '
                            . $product->get_attribute( $product_attribute['name'] ) . ' ';
                    }
                }

            }
        }

        return $products_prompt;
    }


    /**
     * Get batch embeddings
     *
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function get_batch_embeddings( $chunks ) {
        $open_ai = $this->create_open_ai_client();
        $all_embeddings = [];
        $options = Settings::get_instance()->options;

        foreach ( $chunks as $batch ) {
            $texts = array_map( fn($p) => json_encode( $p ), $batch );

            $chunk_embedding_json = $open_ai->embeddings( [
                'model' => esc_attr( $options['open_ai_embedding_model'] ),
                'input' => $texts
            ] );

            $chunk_embedding = json_decode( $chunk_embedding_json, true );
            if ( isset( $chunk_embedding['data'] ) ) {
                $embeddings = $chunk_embedding['data'];
                foreach ( $batch as $index => $batch_item ) {
                    $all_embeddings[] = [
                        'id' => $batch_item['id'],
                        'embedding' => $embeddings[$index]['embedding'] ?? []
                    ];
                }
            }
        }

        return $all_embeddings;
    }

    /**
     * Get products data and current product snapshot
     *
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function get_products_data() {
        $products = wc_get_products( [ 'status' => 'publish', 'limit' => -1  ] );
        $products_data = [];
        $current_snapshot = [];

        if ( !empty( $products ) ) {
            foreach ( $products as $product ) {
                $product_data = $product->get_data();
                $product_cats = $this->get_product_categories( $product_data['id'] );
                $in_stock = $product_data['stock_status'] === 'instock' ? 'yes' : 'no';
                $on_sale = !empty( $product_data['sale_price'] ) ? 'yes' : 'no';

                $products_array = [
                    'id' => $product_data['id'],
                    'name' => $product_data['name'],
                    'description' => $product_data['description'],
                    'category' => $product_cats,
                    'price' => $product_data['price'] . get_woocommerce_currency_symbol(),
                    'in_stock' => $in_stock,
                    'on_sale' => $on_sale,
                    'link' => get_permalink( $product_data['id'] ),
                ];

                foreach ( $product->get_attributes() as $product_attribute ) {
                    $products_array[wc_get_attribute( $product_attribute['id'] )->name ] = $product->get_attribute( $product_attribute['name'] );
                }

                $current_snapshot[$product_data['id']] = md5( json_encode( $products_array ) );
                $products_data[] = $products_array;
            }
        }

        return [
            'product_data' => $products_data,
            'current_snapshot' => $current_snapshot
        ];
    }

    /**
     * Create embeddings for WooCommerce products
     *
     * @return void
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function create_product_embeddings() {
        $products_data_array = $this->get_products_data();
        $products_data = $products_data_array['product_data'];
        $current_product_snapshot = $products_data_array['current_snapshot'];
        if ( empty( $products_data ) ) { return; }
        $previous_snapshot = get_option( 'mdp_helper_current_products_snapshot', [] );
        $product_embeddings = get_option( 'mdp_helper_product_embeddings', [] );

        if ( empty( $previous_snapshot ) || empty( 'mdp_helper_product_embeddings' ) ) {
            $this->generate_products_embeddings( $products_data );
            update_option( 'mdp_helper_current_products_snapshot', $current_product_snapshot );
            return;
        }

        // Detect changes
        $added = array_diff_key( $current_product_snapshot, $previous_snapshot );
        $deleted = array_diff_key( $previous_snapshot, $current_product_snapshot );
        $changed = [];

        foreach ( $current_product_snapshot as $id => $hash ) {
            if ( isset( $previous_snapshot[$id] ) && $previous_snapshot[$id] !== $hash ) {
                $changed[$id] = $hash;
            }
        }

        // If there are any changes, regenerate embeddings
        if ( !empty( $added ) || !empty( $deleted ) || !empty( $changed ) ) {
            $products_to_update = [];
            foreach ( array_keys( $added + $changed ) as $product_id ) {
                $products_to_update[] = array_values( array_filter( $products_data, fn( $item ) => $item['id'] === $product_id ) )[0];
            }

            if ( !empty( $products_to_update ) ) {
                $updated_embeddings =  $this->generate_products_embeddings( $products_to_update, false );
                update_option( 'mdp_helper_product_embeddings', array_merge( $product_embeddings, $updated_embeddings ) );

                // Save updated snapshot
                update_option( 'mdp_helper_current_products_snapshot', $current_product_snapshot );
            }
        }
    }

    /**
     * Generate WooCommerce products embeddings
     *
     * @param bool $update_option
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function generate_products_embeddings( $products_data, $update_option = true ) {
        if (  !in_array('woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
            return [];
        }

        $product_chunks = array_chunk( $products_data, 20 );
        $embeddings = $this->get_batch_embeddings( $product_chunks );

        if ( $update_option ) {
            update_option( 'mdp_helper_product_embeddings', $embeddings );
        }

        return $embeddings;
    }

    /**
     * Generate embeddings for text prompt
     *
     * @param $text
     * @param $type
     * @param int $chunk_size
     * @param int $overlap
     * @return void
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function generate_text_embeddings( $text, $type, $chunk_size = 500, $overlap = 100 ) {
        $text_hash = get_option( 'mdp_helper_' . $type . '_text_hash', '' );
        $options = Settings::get_instance()->options;

        // Exit if no need to update text embeddings
        if ( !empty( $text_hash ) && $text_hash === md5( $text ) ) {
            return;
        }

        $open_ai = $this->create_open_ai_client();
        $sentences = preg_split( '/(?<=[.!?])\s+/', $text, -1, PREG_SPLIT_NO_EMPTY );
        $chunks = [];
        $current_chunk = "";

        foreach ( $sentences as $sentence ) {
            if ( str_word_count( $current_chunk . " " . $sentence ) > $chunk_size ) {
                $chunks[] = $current_chunk;
                $current_chunk = implode( " ", array_slice( explode( " ", $current_chunk ), -$overlap ) ); // Keep overlap
            }

            $current_chunk .= " " . $sentence;
        }

        if ( !empty( $current_chunk ) ) {
            $chunks[] = $current_chunk;
        }

        $text_embeddings = [];

        foreach ( $chunks as $chunk ) {
            $chunk_data_json = $open_ai->embeddings( [
                'model' => esc_attr( $options['open_ai_embedding_model'] ),
                'input' => $chunk
            ] );

            $chunk_data = json_decode( $chunk_data_json, true );
            if ( $chunk_data['data'][0]['embedding'] ) {
                $text_embeddings[] = [
                    'text' => $chunk,
                    'embedding' => $chunk_data['data'][0]['embedding']
                ];
            }
        }

        update_option( 'mdp_helper_' . $type . '_text_chunks', $text_embeddings );
        update_option( 'mdp_helper_' . $type . '_text_hash', md5( $text ) );
    }

    /**
     * Returns PDF file content
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function get_pdf_file_content() {
        $options = Settings::get_instance()->options;
        $file = esc_attr( $options['open_ai_pdf_file'] );
        $text = '';

        $parser = new Parser();
        if ( !empty( $file ) ) {
            $pdf = $parser->parseFile( wp_get_upload_dir()['baseurl'] . '/helper/open_ai_pdf_file/' . $file );
            $text = wp_kses_post( $pdf->getText() );
        }

        return $text;
    }

    /**
     * Creates Open Ai object
     *
     * @return OpenAi
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function create_open_ai_client() {
        $options = Settings::get_instance()->options;

        $enabled_additional_keys = !empty( $options['open_ai_add_additional_keys'] ) &&
            $options['open_ai_add_additional_keys'] === 'on';
        $open_ai_keys = Caster::get_instance()->get_all_repeater_data( 10, 'open_ai_api_key_' );

        /** If keys not provided return */
        if ( empty( $options['open_ai_api_key'] ) ) { return null; }

        if ( $enabled_additional_keys && !empty( $open_ai_keys ) ) {
            $open_ai_keys[] = $options['open_ai_api_key'];
        }

        $open_ai_key = $enabled_additional_keys && !empty( $open_ai_keys ) ?
            esc_attr( $open_ai_keys[rand( 0, count( $open_ai_keys ) - 1 ) ] ) :
            esc_attr( $options['open_ai_api_key'] );

        $open_ai_client = new OpenAi( $open_ai_key );
        $open_ai_client->setAssistantsBetaVersion( 'v2' );

        return $open_ai_client;
    }

    /**
     * Calculates the cosine similarity between two vectors.
     *
     * Cosine similarity measures the cosine of the angle between two non-zero vectors
     * in an n-dimensional space. It is used to determine how similar two vectors are
     * regardless of their magnitude.
     *
     * @param array $vector1 The first vector (embedding) as an array of floats.
     * @param array $vector2 The second vector (embedding) as an array of floats.
     * @return float The cosine similarity score between -1 (opposite) and 1 (identical).
     *               A score closer to 1 indicates higher similarity.
     */
    private function cosine_similarity( $vector1, $vector2 ) {
        $dotProduct = 0;
        $normA = 0;
        $normB = 0;

        for ( $i = 0; $i < count( $vector1 ); $i++ ) {
            $dotProduct += $vector1[$i] * $vector2[$i];
            $normA += pow( $vector1[$i], 2 );
            $normB += pow( $vector2[$i], 2 );
        }

        return $dotProduct / ( sqrt( $normA ) * sqrt( $normB ) );
    }

    /**
     * Find the best WooCommerce products match for a query
     *
     * @return array|null
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function get_best_match_products( $query_embedding, $product_embeddings, $limit = 10 ) {

        if ( empty( $query_embedding ) ) {
            return null;
        }

        $products_data = $this->get_products_data()['product_data'];
        $scores = [];
        $best_match_products = [];

        foreach ( $product_embeddings as $product_embedding ) {
            $score = $this->cosine_similarity( $query_embedding, $product_embedding['embedding'] );
            $scores[] = ['id' => $product_embedding['id'], 'score' => $score];
        }

        // Sort by highest score
        usort( $scores, fn( $a, $b ) => $b['score'] <=> $a['score'] );

        $best_match_products_embeddings = array_slice( $scores, 0, $limit );

        foreach ( $best_match_products_embeddings as $best_match_products_embedding ) {
            $best_match_products[] = array_filter( $products_data, fn( $item ) => $item['id'] === $best_match_products_embedding['id'] );
        }

        // Return the top N products
        return [
            'products' => $best_match_products,
            'highest_score' => max( array_column( $best_match_products_embeddings, 'score' ) )
        ];

    }

    /**
     * Find the best WooCommerce products match for a text prompt
     *
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function get_best_text_match( $embedding, $query_embedding, $highest_score ) {
        $best_match = '';
        foreach ( $embedding as $entry ) {
            $stored_embedding = $entry['embedding'];
            $score = $this->cosine_similarity( $stored_embedding, $query_embedding );

            if ( $score > $highest_score ) {
                $highest_score = $score;
                $best_match = $entry['text'];
            }
        }

        return [
            'highest_score' => $highest_score,
            'best_match' => $best_match
        ];
    }

    /**
     * Find the best match for a query through all embeddings
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function search_best_match( $query ) {
        $open_ai = $this->create_open_ai_client();
        $options = Settings::get_instance()->options;

        // Convert query to an embedding
        $query_embedding_json = $open_ai->embeddings([
            'model' => esc_attr( $options['open_ai_embedding_model'] ),
            'input' => $query
        ]);

        $query_embedding = json_decode( $query_embedding_json, true );

        if ( empty( $query_embedding['data'][0]['embedding'] ) ) {
            return '';
        }

        $product_embeddings = get_option( 'mdp_helper_product_embeddings' );
        $pdf_embeddings = get_option( 'mdp_helper_pdf_text_chunks' );
        $custom_embeddings = get_option( 'mdp_helper_custom_text_chunks' );
        $post_embeddings = get_option( 'mdp_helper_post_text_chunks' );

        $best_match = null;
        $highest_score = -1;

        if ( !empty( $product_embeddings ) && in_array( 'woocommerce_products', $options['open_ai_prompt_type'] ) ) {
            $best_match_products = $this->get_best_match_products( $query_embedding['data'][0]['embedding'], $product_embeddings );
            $best_match = json_encode( $best_match_products['products'] );
            $highest_score = $best_match_products['highest_score'];
        }

        if ( !empty( $pdf_embeddings ) && in_array( 'pdf_file', $options['open_ai_prompt_type'] ) ) {
            $best_text_match = $this->get_best_text_match( $pdf_embeddings, $query_embedding['data'][0]['embedding'], $highest_score );
            if ( !empty( $best_text_match['best_match'] ) ) {
                $highest_score = $best_text_match['highest_score'];
                $best_match = $best_text_match['best_match'];
            }
        }


        if ( !empty( $custom_embeddings ) && in_array( 'custom', $options['open_ai_prompt_type'] ) ) {
            $best_text_match = $this->get_best_text_match( $custom_embeddings, $query_embedding['data'][0]['embedding'], $highest_score );
            if ( !empty( $best_text_match['best_match'] ) ) {
                $highest_score = $best_text_match['highest_score'];
                $best_match = $best_text_match['best_match'];
            }
        }

        if ( !empty( $post_embeddings ) && in_array( 'post_content', $options['open_ai_prompt_type'] ) ) {
            $best_text_match = $this->get_best_text_match( $post_embeddings, $query_embedding['data'][0]['embedding'], $highest_score );
            if ( !empty( $best_text_match['best_match'] ) ) {
                $highest_score = $best_text_match['highest_score'];
                $best_match = $best_text_match['best_match'];
            }
        }

        return $best_match ?? '';
    }

    /**
     * Create embeddings for prompt.
     *
     * @return void
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function create_prompt_embeddings() {
        $options = Settings::get_instance()->options;

        if ( in_array( 'custom', $options['open_ai_prompt_type'] ) ) {
            $this->generate_text_embeddings( esc_html( $options['open_ai_prompt'] ), 'custom' );
        }

        if ( in_array( 'woocommerce_products', $options['open_ai_prompt_type'] ) ) {
            $this->create_product_embeddings();
        }

        if ( in_array( 'post_content', $options['open_ai_prompt_type'] )  ) {
            $this->generate_text_embeddings( $this->get_post_content(), 'post' );
        }

        if ( in_array( 'pdf_file', $options['open_ai_prompt_type'] ) ) {
            $this->generate_text_embeddings( $this->get_pdf_file_content(), 'pdf' );
        }
    }

    /**
     * Get prompt based on selected context.
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     * @access public
     */
    public function get_result_prompt( $question ) {
        $options = Settings::get_instance()->options;
        $prompt = '';
        $objective = "\n\nObjective:" . esc_html( $options['open_ai_objective'] );
        $notice = "\n\nNotice:" . esc_html( $options['open_ai_notice'] );

        if ( $options['open_ai_enable_embeddings'] === 'on' ) {
            $this->create_prompt_embeddings();
            $prompt = $this->search_best_match( $question );
        } else {
            if ( in_array( 'custom', $options['open_ai_prompt_type'] ) ) {

                $description = esc_html( $options['open_ai_prompt'] );

                $prompt .= $description;

            }

            if ( in_array( 'woocommerce_products', $options['open_ai_prompt_type'] ) ) {
                $prompt .= $this->get_products_prompt();
            }

            if ( in_array( 'post_content', $options['open_ai_prompt_type'] )  ) {
                $prompt .= $this->get_post_content();
            }

            if ( in_array( 'pdf_file', $options['open_ai_prompt_type'] ) ) {
                $prompt .= $this->get_pdf_file_content();
            }
        }

        $result_prompt = $prompt . "\n\n" . $objective . "\n\n" . $notice . "\n\nQuestion:" . $question . "\n\nAnswer:\n\n";
        return trim( preg_replace( '/\s\s+/', ' ', str_replace( "\n", " ", $result_prompt ) ) );

    }

    /**
     * @throws \Exception
     */
    public function stream_bot_response( $message ) {
        $current_bot_options = Caster::get_instance()->get_current_personality_options();

        if ( $current_bot_options['type'] !== 'assistant' ) {
            OpenAiBot::get_instance()->get_stream_bot_response( $message );
        } else {
            $session_id = sanitize_text_field( $_GET['mdp_helper_session_id'] );
            OpenAiBot::get_instance()->stream_assistant_response( $message, $current_bot_options, $session_id );
        }
    }

    /**
     * Stream bot response.
     *
     * @return void
     * @throws \Exception
     * @since 1.0.0
     * @access public
     */
    public function get_stream_bot_response( $question ) {
        $open_ai = $this->create_open_ai_client();
        $options = Settings::get_instance()->options;
        $model = esc_attr( $options['open_ai_model'] );

        if ( empty( $open_ai ) ) {
            echo "event: stop\n";
            echo "data: stopped\n\n";
        }

        $result_prompt = $this->get_result_prompt( $question );

        $open_ai->chat( [
            'model' => $model,
            "stream" => true,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => $result_prompt
                ]
            ],
            'temperature' => (int)esc_attr( $options['open_ai_temperature'] ),
            'max_tokens' => (int)esc_attr( $options['open_ai_max_tokens'] ),
            'top_p' => 1,
            'frequency_penalty' => 0,
            'presence_penalty' => 0,
            'stop' => [
                "\nNote:",
                "\nQuestion:"
            ]
        ], function ($ch, $data) {
            $deltas = explode( "\n", $data );
            foreach ( $deltas as $delta ) {

                if ( strpos( $delta, "data: " ) !== 0 ) {
                    continue;
                }

                $json = json_decode( substr( $delta, 6 ) );
                if ( isset( $json->choices[0]->delta ) ) {
                    $content = $json->choices[0]->delta->content ?? "";
                } elseif ( isset( $json->error->message) ) {
                    $content = $json->error->message;
                } elseif ( trim( $delta ) === "data: [DONE]" ) {
                    $content = '';
                } else {
                    $content = 'Sorry but i dont know how to answer this.';
                }

                echo "data: " . $content . "\n\n";
                flush();
            }

            if ( connection_aborted() ) { return 0; }

            return strlen( $data );
        }  );
    }

    /**
     * Return respond from Open AI
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function get_open_ai_bot_result( $question ) {
        $options = Settings::get_instance()->options;
        $model = esc_attr( $options['open_ai_model'] );

        $open_ai = $this->create_open_ai_client();

        $chat_completions_models = [
            'gpt-3.5-turbo',
            'gpt-3.5-turbo-16k',
            'gpt-3.5-turbo-1106',
            'gpt-4',
            'gpt-4-0125-preview',
            'gpt-4-1106-preview',
            'gpt-4o',
            'gpt-4o-mini',
            'gpt-4-turbo',
        ];

        if ( empty( $open_ai ) ) {
            return current_user_can( 'editor' ) || current_user_can( 'administrator' ) ?
                esc_html__( 'Something went wrong! Please check your Open AI API key!' ) :
                esc_html__( 'Something went wrong. Try again later.', 'helper' );
        }

        $result_prompt = $this->get_result_prompt( $question );

        if ( in_array( $model, $chat_completions_models ) ) {
            $complete = json_decode( $open_ai->chat( [
                'model' => $model,
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $result_prompt
                    ]
                ],
                'temperature' => (int)esc_attr( $options['open_ai_temperature'] ),
                'max_tokens' => (int)esc_attr( $options['open_ai_max_tokens'] ),
                'top_p' => 1,
                'frequency_penalty' => 0,
                'presence_penalty' => 0,
                'stop' => [
                    "\nNote:",
                    "\nQuestion:"
                ]
            ] ) );
        } else {
            $complete = json_decode( $open_ai->completion( [
                'model' => $model,
                'prompt' => $result_prompt,
                'temperature' => (int)esc_attr( $options['open_ai_temperature'] ),
                'max_tokens' => (int)esc_attr( $options['open_ai_max_tokens'] ),
                'top_p' => 1,
                'frequency_penalty' => 0,
                'presence_penalty' => 0,
                'stop' => [
                    "\nNote:",
                    "\nQuestion:"
                ]
            ] ) );
        }

        if ( isset( $complete->choices[0]->text ) && $model !== 'gpt-3.5-turbo' && $model !== 'gpt-3.5-turbo-16k' && $model !== 'gpt-3.5-turbo-1106' ) {
            $text = str_replace( "\\n", "\n", $complete->choices[0]->text );
        } elseif( isset( $complete->choices[0]->message->content ) ) {
            $text = str_replace( "\\n", "\n", $complete->choices[0]->message->content );
        } elseif ( isset( $complete->error->message ) ) {
            $text = current_user_can( 'editor' ) || current_user_can( 'administrator' ) ?
                $complete->error->message :
                esc_html__( 'Something went wrong. Try again later.', 'helper' );
        } else {
            $text = esc_html__( "Sorry, but I don't know how to answer that.", 'helper' );
        }


        return nl2br( $text );
    }

    /**
     * Returns list of created assistants
     *
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function get_assistants_list() {
        $open_ai = $this->create_open_ai_client();

        if ( empty( $open_ai ) ) { return []; }

        return json_decode( $open_ai->listAssistants() )->data ?? [];
    }

    /**
     * Creates assistant thread
     *
     * @return array
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function create_assistant_thread() {
        $open_ai = $this->create_open_ai_client();
        if ( empty( $open_ai ) ) { return ''; }
        return json_decode( $open_ai->createThread() )->id;
    }

    /**
     * Removes assistant thread
     *
     * @return bool|string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function delete_assistant_thread( $thread_id ) {
        $open_ai = $this->create_open_ai_client();
        if ( empty( $open_ai ) ) { return ''; }
        return $open_ai->deleteThread( $thread_id );
    }

    /**
     * Filters assistant messages
     *
     * @return bool|string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    private function filter_thread_messages( $messages, $run_id ) {
        $filtered_messages = [];

        foreach ( $messages as $message ) {
            if ( $message->run_id === $run_id && $message->role === "assistant" ) {
                $filtered_messages[] = $message;
            }
        }

        return array_pop( $filtered_messages );
    }


    /**
     * @throws \Exception
     */
    public function stream_assistant_response($message, $bot_options, $session_id ) {
        $options = Settings::get_instance()->options;
        $open_ai = $this->create_open_ai_client();

        $assistant_id = esc_attr( $bot_options['assistant'] );

        /** If keys not provided return */
        if ( empty( $options['open_ai_api_key'] ) ) {
            echo "event: stop\n";
            echo "data: stopped\n\n";
        }

        // Create Thread
        $thread_id = get_option( $session_id );

        if ( empty( $thread_id ) ) {
            echo "event: stop\n";
            echo "data: stopped\n\n";
        }

        // Create thread message
        $open_ai->createThreadMessage( $thread_id, [
            'role' => 'user',
            'content' => $message
        ] );

        // Create Thread
        $thread_id = get_option( $session_id );

        if ( empty( $thread_id ) ) {
            echo "event: stop\n";
            echo "data: stopped\n\n";
        }

        // Create thread message
        $open_ai->createThreadMessage( $thread_id, [
            'role' => 'user',
            'content' => $message
        ] );

        // Create run
        $open_ai->createRun( $thread_id, [
            'assistant_id' => $assistant_id,
            'stream' => true,
        ], function ( $curl_info, $data ) {
            $thread_data = explode( "\ndata:", $data );
            $thread_status = substr( $thread_data[0], 7 );
            $content = '';
            if ( $thread_status === 'thread.message.delta' ) {
                foreach ( $thread_data as $thread_data_line ) {
                    if ( !strpos( $thread_data_line, 'event: ' ) ) {
                        $message_delta = json_decode( $thread_data_line );
                        if ( !empty( $message_delta ) ) {
                            $content = "data: " . $message_delta->delta->content[0]->text->value . "\n\n";
                        }
                    }
                }
            }
            echo $content;
            flush();

            if ( connection_aborted() ) { return 0; }

            return strlen( $data );
        } );

    }

    /**
     * Returns response from assistant
     *
     * @return string
     * @throws \Exception
     * @since 1.0.0
     * @access private
     */
    public function get_assistant_response( $message, $bot_options, $session_id ) {
        $options = Settings::get_instance()->options;

        /** If keys not provided return */
        if ( empty( $options['open_ai_api_key'] ) ) {
            return current_user_can( 'editor' ) || current_user_can( 'administrator' ) ?
                esc_html__( 'Something went wrong! Please check your Open AI API key!' ) :
                esc_html__( 'Something went wrong. Try again later.', 'helper' );
        }

        $open_ai = $this->create_open_ai_client();

        $assistant_id = esc_attr( $bot_options['assistant'] );

        if ( empty( $assistant_id ) ) {
            return current_user_can( 'editor' ) || current_user_can( 'administrator' ) ?
                esc_html__( 'Something went wrong! Please select Open AI assistant!' ) :
                esc_html__( 'Something went wrong. Try again later.', 'helper' );
        }

        // Create Thread
        $thread_id = get_option( $session_id );

        if ( empty( $thread_id ) ) {
            return esc_html__( 'Something went wrong. Try again later.', 'helper' );
        }

        // Create thread message
        $open_ai->createThreadMessage( $thread_id, [
            'role' => 'user',
            'content' => $message
        ] );

        // Create run
        $run_json = $open_ai->createRun( $thread_id, [
            'assistant_id' => $assistant_id
        ] );

        $run_data = json_decode( $run_json );

        // Create Response
        $response = json_decode( $open_ai->retrieveRun( $thread_id, $run_data->id ) );

        // Polling to retrieve run status
        while ( $response->status === 'in_progress' || $response->status === 'queued' ) {
            sleep( 5 );
            $response = json_decode( $open_ai->retrieveRun( $thread_id, $run_data->id ) );
        }

        // Send error if response failed
        if ( $response->status === 'failed' ) {
            return current_user_can( 'editor' ) || current_user_can( 'administrator' ) ?
                $response->last_error->message :
                esc_html__( 'Something went wrong. Try again later.', 'helper' );
        }

        // Get messages list
        $messages_list = json_decode( $open_ai->listThreadMessages( $thread_id ) );
        $messages = $messages_list->data;
        $result_message_object = $this->filter_thread_messages( $messages, $run_data->id );
        $result_message = $result_message_object->content[0]->text->value;
        $result_message = preg_replace( '/【.*?†.*?】/', '', $result_message );

        return nl2br( $result_message );
    }

    /**
     * Main Caster Instance.
     * Insures that only one instance of OpenAiBot exists in memory at any one time.
     *
     * @static
     * @since 1.0.0
     * @access public
     *
     * @return OpenAiBot
     **/
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

            self::$instance = new self;

        }

        return self::$instance;

    }
}
