<?php
/**
 * Helper
 * Create a chatbot with AI features for your website.
 * Exclusively on https://1.envato.market/helper
 *
 * @encoding        UTF-8
 * @version         1.1.2
 * @copyright       (C) 2018 - 2023 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Cherviakov Vlad (vladchervjakov@gmail.com), Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmytro Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\Helper;

use Merkulove\Helper\Unity\Plugin;
use Merkulove\Helper\Unity\Settings;
use Merkulove\Helper\Unity\TabGeneral;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * @package Merkulove\Helper
 **/
final class TabWelcomePopup {

	/**
	 * Controls.
	 * @return array[]
	 */
	public static function setTab(): array {

		$options = Settings::get_instance()->options;

		$tabs = Plugin::get_tabs();

        $offset = 4;
        $tabs = array_slice( $tabs, 0, $offset, true ) +
            ['welcome_popup' => [
                'enabled'       => true,
                'class'         => TabGeneral::class, // Handler
                'label'         => esc_html__( 'Welcome popup', 'helper' ),
                'title'         => esc_html__( 'Welcome popup', 'helper' ),
                'show_title'    => true,
                'icon'          => 'tab',
                'fields'        => []
            ] ] +
            array_slice( $tabs, $offset, NULL, true );

		$fields = array_merge(
			self::fields_chat_container( $options ),
            self::fields_animation( $options ),
            self::fields_close_button( $options )
        );

		$tabs[ 'welcome_popup' ][ 'fields' ] = $fields;
		Plugin::set_tabs( $tabs );
		Settings::get_instance()->get_options();

		return $tabs;

	}

    /**
     * Fields for header close button.
     *
     * @param $options
     *
     * @return array
     */
    private static function fields_close_button( $options ): array {

        $slug = 'welcome_popup_close_button';

        return [

            $slug . '_divider' => [
                'type' => 'divider',
            ],

            $slug . '_title' => [
                'type'              => 'header',
                'label'             => esc_html__( 'Welcome popup close button settings', 'helper' ),
                'description'       => esc_html__( 'Display and style settings for the close button', 'helper' ),
            ],

            $slug . '_show' => [
                'type' => 'switcher',
                'label' => esc_html__( 'Close button', 'helper' ),
                'description' => esc_html__( 'Display close button.', 'helper' ),
                'default' => 'on',
            ],

            $slug . '_margin' => [
                'type' => 'sides',
                'label' => esc_html__( 'Close button margin', 'helper' ),
                'description' => esc_html__( 'Set close button margin.', 'helper' ),
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 15,
                    'left' => 0,
                ],
            ],

            $slug . '_padding' => [
                'type' => 'sides',
                'label' => esc_html__( 'Close button padding', 'helper' ),
                'description' => esc_html__( 'Set close button padding.', 'helper' ),
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
            ],

            $slug . '_icon' => [
                'type'              => 'icon',
                'label'             => esc_html__( 'Close icon', 'helper' ),
                'description'       => esc_html__( 'Set close chat icon.', 'helper' ),
                'default'           => 'helper/close.svg',
                'meta'              => [
                    'helper.json',
                    'font-awesome.json',
                    'material.json'
                ]
            ],

            $slug . '_size' => [
                'type' => 'slider',
                'label' => esc_html__( 'Icon size', 'helper' ),
                'description' => wp_sprintf(
                /* translators: %s: icon size */
                    esc_html__( 'Current icon size: %s pixels.', 'helper' ),
                    '<strong>' . esc_attr( ( ! empty( $options[ $slug . '_size' ] ) ) ?
                        $options[ $slug . '_size' ] : '22' ) . '</strong>'
                ),
                'default' => 22,
                'min' => 1,
                'max' => 40,
                'step' => 1,
                'discrete' => false,
            ],

            $slug . '_color' => [
                'type' => 'colorpicker',
                'label' => esc_html__( 'Icon color', 'helper' ),
                'description' => esc_html__( 'Set close icon color.', 'helper' ),
                'default' => '#6a52e2',
            ],

            $slug . '_color_hover' => [
                'type' => 'colorpicker',
                'label' => esc_html__( 'Hover Icon color', 'helper' ),
                'description' => esc_html__( 'Set close icon color on hover.', 'helper' ),
                'default' => '#4f32e6',
            ],

        ];

    }

	/**
	 * Fields for chat popup container
	 *
	 * @param $options
	 *
	 * @return array
	 */
	private static function fields_chat_container( $options ): array {

		$slug = 'welcome_popup';

		return array_merge(

			[
                $slug . '_enable' => [
                    'type'              => 'switcher',
                    'label'             => esc_html__( 'Enable welcome popup', 'helper' ),
                    'placeholder'       => esc_html__( 'Enable welcome popup', 'helper' ),
                    'description'       => esc_html__( 'Enables welcome popup on page load', 'helper' ),
                    'default'           => 'off',
                ],

                $slug . '_open_delay' => [
                    'type'              => 'slider',
                    'label'             => esc_html__( 'Open delay', 'helper' ),
                    'placeholder'       => esc_html__( 'Open delay', 'helper' ),
                    'description'       => wp_sprintf(
                        esc_html__( 'Open after %s seconds.', 'helper' ),
                        '<strong>' . esc_attr( ( ! empty( $options[ $slug . '_open_delay' ] ) ) ?
                            $options[ $slug . '_open_delay' ] : '0.5' ) . '</strong>'
                    ),
                    'default'           => 0.5,
                    'min'               => 0,
                    'max'               => 30,
                    'step'              => 0.1,
                    'discrete'          => false,
                ],

                $slug . '_content' => [
                    'type'              => 'editor',
                    'label'             => esc_html__( 'Welcome popup content', 'helper' ),
                    'description'       => esc_html__( 'Welcome popup content', 'helper' ),
                    'default'           => wp_kses_post( '<p>Hi!👋<p> <p>Helper here ready to help you!<p>' ),
                    'attr'              => [
                        'textarea_rows' => ''
                    ]
                ],


                $slug . '_divider' => [
					'type' => 'divider',
				],

				$slug . '_margin' => [
					'type'			  => 'sides',
					'label'			  => esc_html__( 'Margin', 'helper' ),
					'description'	  => esc_html__( 'Set welcome popup margin.', 'helper' ),
					'default'         => [
						'top' => 0,
						'right' => 20,
						'bottom' => 20,
						'left' => 20,
					],
				],

                $slug . '_padding' => [
                    'type'			  => 'sides',
                    'label'			  => esc_html__( 'Padding', 'helper' ),
                    'description'	  => esc_html__( 'Set welcome popup padding', 'helper' ),
                    'default'         => [
                        'top' => 20,
                        'right' => 0,
                        'bottom' => 0,
                        'left' => 20,
                    ],
                ],

				$slug . '_max_width' => [
					'type'              => 'switcher',
					'label'             => esc_html__( 'Max width 100%', 'helper' ),
					'placeholder'       => esc_html__( 'Max width 100%', 'helper' ),
					'description'       => esc_html__( 'Max width and margins equals 100% of screen', 'helper' ),
					'default'           => 'on',
				],

				$slug . '_width' => [
					'type'              => 'custom_type',
					'render'            => [ Config::get_instance(), 'render_measures_slider' ],
					'label'             => esc_html__( 'Width', 'helper' ),
					'description'       => wp_sprintf(
						esc_html__( 'Set bot container width. Current width is %s', 'helper' ),
						'<strong>' . esc_html( ( ! empty( $options[$slug . '_width'] ) ) ? $options[$slug . '_width'] : '350' ) . '</strong>'
					),
					'default'           => 350,
					'min'               => 1,
					'max'               => 1000,
					'step'              => 1,
					'discrete'          => false,
				],

				$slug . '_height' => [
					'type'              => 'custom_type',
					'render'            => [ Config::get_instance(), 'render_measures_slider' ],
					'label'             => esc_html__( 'Height', 'helper' ),
					'show_label'        => true,
					'description'       => wp_sprintf(
						esc_html__( 'Set bot container height. Current height is %s', 'helper' ),
						'<strong>' . esc_html( ( ! empty( $options[$slug . '_height'] ) ) ? $options[$slug . '_height'] : '100' ) . '</strong>'
					),
					'default'           => 100,
					'min'               => 1,
					'max'               => 1000,
					'step'              => 1,
					'discrete'          => false,
				],

				$slug . '_background_color' => [
					'type' => 'colorpicker',
					'label' => esc_html__( 'Background color', 'helper' ),
					'description' => esc_html__( 'Set chat window background color.', 'helper' ),
					'default' => '#ffffff',
				],

			],

			Config::get_instance()->border_controls(
				$slug,
				[
					'border_style' => 'none',
					'border_color' => '#4f32e6',
					'border_width' => [
						'top' => 1,
						'right' => 1,
						'bottom' => 1,
						'left' => 1,
					],
					'border_radius' => [
						'top' => 20,
						'right' => 20,
						'bottom' => 20,
						'left' => 20,
					],
				]
			),

			[

				$slug . '_box_shadow' => [
					'type' => 'select',
					'label' => esc_html__( 'Box shadow', 'helper' ),
					'description' => esc_html__( 'Set chat window box shadow.', 'helper' ),
					'default' => 'outside',
					'options' => [
						'none' => esc_html__( 'None', 'helper' ),
						'outside' => esc_html__( 'Outside box-shadow', 'helper' ),
						'inside' => esc_html__( 'Inside box-shadow', 'helper' ),
					],
				],

				$slug . '_box_shadow_offset' => [
					'type' => 'sides',
					'label' => esc_html__( 'Box shadow offset', 'helper' ),
					'description' => esc_html__( 'Set chat window box shadow offset.', 'helper' ),
					'default' => [
						'top' => '0',
						'right' => '-5',
						'bottom' => '20',
						'left' => '0',
					],
					'labels' => [
						'top' => esc_html__( 'X-offset', 'helper' ),
						'right' => esc_html__( 'Y-offset', 'helper' ),
						'bottom' => esc_html__( 'Blur', 'helper' ),
						'left' => esc_html__( 'Spread', 'helper' ),
					],
				],

				$slug . '_box_shadow_color' => [
					'type' => 'colorpicker',
					'label' => esc_html__( 'Box shadow color', 'helper' ),
					'description' => esc_html__( 'Set chat window box shadow color.', 'helper' ),
					'default' => 'rgba(0,0,0,.08)',
				],

			]

		);

	}

	/**
	 * Fields animation
	 * @param $options
	 *
	 * @return array[]
	 */
	private static function fields_animation( $options ): array {

		$slug = 'welcome_popup';

		return [

			$slug . '_animation_divider' => [
				'type' => 'divider',
			],

			$slug . '_animation' => [
				'type'              => 'select',
				'label'             => esc_html__( 'Welcome popup appearance animation', 'helper' ),
				'description'       => esc_html__( 'Set welcome popup appearance animation', 'helper' ),
				'default'           => 'slide_up',
				'options'           => [
					'none' => esc_html__( 'None', 'helper' ),
					'grow' => esc_html__( 'Grow', 'helper' ),
					'shrink' => esc_html__( 'Shrink', 'helper' ),
					'slide_up' => esc_html__( 'Slide up', 'helper' ),
					'slide_down' => esc_html__( 'Slide down', 'helper' ),
					'fade' => esc_html__( 'Fade', 'helper' )
				],
                'attr' => [
                    'class' => 'mdp-helper-animation-control'
                ]
			],

			$slug . '_animation_duration' => [
				'type'              => 'slider',
				'label'             => esc_html__( 'Animation duration', 'helper' ),
				'description'       => wp_sprintf(
					esc_html__( 'Set welcome popup appearance animation duration. Current animation duration is %s seconds.', 'helper' ),
					'<strong>' . esc_html( ( ! empty( $options[ $slug . '_animation_duration' ] ) ) ? $options[ $slug . '_animation_duration' ] : '1' ) . '</strong>'
				),
				'default'           => 1,
				'min'               => 0.1,
				'max'               => 5,
				'step'              => 0.1,
				'discrete'          => false,
                'attr'              => [
                    'class' => '
                       mdp-helper-animation-control 
                       mdp-helper-welcome-popup-animation 
                       mdp-helper-tab-animation-type-grow
                       mdp-helper-tab-animation-type-shrink 
                       mdp-helper-tab-animation-type-slide_up
                       mdp-helper-tab-animation-type-slide_down
                       mdp-helper-tab-animation-type-fade'
                ]

			],

			$slug . '_animation_delay' => [
				'type'              => 'slider',
				'label'             => esc_html__( 'Animation delay', 'helper' ),
				'description'       => wp_sprintf(
					esc_html__( 'Set welcome popup appearance animation delay. Current animation delay is %s seconds.', 'helper' ),
					'<strong>' . esc_html( ( ! empty( $options[ $slug . '_animation_delay' ] ) ) ? $options[ $slug . '_animation_delay' ] : '0' ) . '</strong>'
				),
				'default'           => 0,
				'min'               => 0,
				'max'               => 5,
				'step'              => 0.1,
				'discrete'          => false,
                'attr'              => [
                    'class' => '
                       mdp-helper-animation-control 
                       mdp-helper-welcome-popup-animation 
                       mdp-helper-tab-animation-type-grow
                       mdp-helper-tab-animation-type-shrink 
                       mdp-helper-tab-animation-type-slide_up
                       mdp-helper-tab-animation-type-slide_down
                       mdp-helper-tab-animation-type-fade'
                ]
			],

		];

	}

}

