<?php
/**
 * The file that defines the core plugin api class
 *
 * A class definition that includes api's endpoints and functions used across the plugin
 *
 * @link       https://wpswings.com/
 * @since      1.0.0
 *
 * @package    Ultimate Woocommerce Gift Cards
 * @subpackage Ultimate Woocommerce Gift Cards/package/rest-api
 */

/**
 * The core plugin  api class.
 *
 * This is used to define internationalization, api-specific hooks, and
 * endpoints for plugin.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Ultimate Woocommerce Gift Cards
 * @subpackage Ultimate Woocommerce Gift Cards/package/rest-api
 * @author     WP Swings <webmaster@wpswings.com>
 */
class Giftcard_For_Woocommerce_Rest_Api extends WP_REST_Controller {

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin api.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the merthods, and set the hooks for the api and
	 *
	 * @since    1.0.0
	 * @param   string $plugin_name    Name of the plugin.
	 * @param   string $version        Version of the plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;

	}

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'api/v1';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $base_url = '/giftcard/';

	/**
	 * Define endpoints for the plugin.
	 *
	 * Uses the Giftcard_For_Woocommerce_Rest_Api class in order to create the endpoint
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	public function wps_uwgc_add_endpoint() {
		// for balance fetch of coupons.
		register_rest_route(
			$this->namespace,
			$this->base_url . 'coupon-details',
			array(
				'args'                => array(
					'consumer_key'    => array(
						'description' => __( 'Merchant Consumer Key.', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'consumer_secret' => array(
						'description' => __( 'Merchant Consumer Secret', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'coupon_code' => array(
						'description' => __( 'Gift Card Coupon Code', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'context'         => array(
						'default' => 'view',
					),
				),
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'wps_uwgc_giftcard_fetch_details' ),
				'permission_callback' => array( $this, 'wps_uwgc_get_permission_check' ),
			)
		);
		// For redeem coupon amount.
		register_rest_route(
			$this->namespace,
			$this->base_url . 'redeem-coupon',
			array(
				'args'                => array(
					'consumer_key'    => array(
						'description' => __( 'Merchant Consumer Key.', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'consumer_secret' => array(
						'description' => __( 'Merchant Consumer Secret', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'coupon_code' => array(
						'description' => __( 'Gift Card Coupon Code', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'redeem_amount' => array(
						'description' => __( 'Gift Card Redeem Amount', 'giftware' ),
						'type'        => 'number',
						'required'    => true,
					),
				),
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'wps_uwgc_giftcard_redeem' ),
				'permission_callback' => array( $this, 'wps_uwgc_get_permission_check' ),
			),
		);

		// for recharging and updating the coupon details.
		register_rest_route(
			$this->namespace,
			$this->base_url . 'recharge-coupon',
			array(
				'args'                => array(
					'consumer_key'       => array(
						'description' => __( 'Merchant Consumer Key.', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'consumer_secret'    => array(
						'description' => __( 'Merchant Consumer Secret', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'coupon_code' => array(
						'description' => __( 'Gift Card Coupon Code', 'giftware' ),
						'type'        => 'string',
						'required'    => true,
					),
					'recharge_amount' => array(
						'description' => __( 'Gift Card Recharge Amount', 'giftware' ),
						'type'        => 'number',
						'required'    => true,
					),
					'coupon_expiry'   => array(
						'description' => __( 'Gift Card Coupon Expiry', 'giftware' ),
						'type'        => 'number',
						'required'    => false,
					),
					'usage_count'     => array(
						'description' => __( 'Gift Card Coupon Usage count', 'giftware' ),
						'type'        => 'number',
						'required'    => false,
					),
					'usage_limit'     => array(
						'description' => __( 'Gift Card Coupon Usage limit', 'giftware' ),
						'type'        => 'number',
						'required'    => false,
					),
				),
				'methods'             => WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'wps_uwgc_giftcard_recharge' ),
				'permission_callback' => array( $this, 'wps_uwgc_get_permission_check' ),
			),
		);
	}

	/**
	 * Begins validation process of api endpoint.
	 *
	 * @param   Array $request    All information related with the api request containing in this array.
	 * @return  Array   $result   return rest response to server from where the endpoint hits.
	 * @since    1.0.0
	 */
	public function wps_uwgc_get_permission_check( $request ) {
		$parameters    = json_decode($request->get_body(), true );
		$rest_api_keys = get_option( 'wps_uwgc_rest_api_keys', '' );
		if ( ! empty( $rest_api_keys ) && is_array( $rest_api_keys ) ) {
			$key    = $parameters['consumer_key'];
			$secret = $parameters['consumer_secret'];
			if ( $key === $rest_api_keys['consumer_key'] && $secret === $rest_api_keys['consumer_secret'] ) {
				return true;
			}
			return new WP_Error( 'rest_forbidden', esc_html__( 'Sorry, your key details are incorrect.', 'giftware' ), array( 'status' => 401 ) );
		}
		return false;
	}

	/**
	 * Returns Coupon details
	 *
	 * @param Array $request All information related with the api request containing in this array.
	 * @return Array
	 */
	public function wps_uwgc_giftcard_fetch_details( $request ) {
		$parameters    = json_decode($request->get_body(), true );
		require_once WPS_UWGC_DIRPATH . 'package/rest-api/version1/class-giftcard-for-woocommerce-rest-api-process.php';
		$wps_uwgc_rest_api_obj     = new Giftcard_For_Woocommerce_Rest_Api_Process();
		$wps_uwgc_resultsdata = $wps_uwgc_rest_api_obj->wps_uwgc_giftcard_fetch_details( $parameters['coupon_code'] );
		if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 200 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata['data'], 200 );
		} else if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 404 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata, 404 ); 
		} else {
			$wps_uwgc_response = new WP_Error( $wps_uwgc_resultsdata );
		}
		return $wps_uwgc_response;
	}

	/**
	 * Returns Updated Coupon details with updated balance after redemption.
	 *
	 * @param Array $request All information related with the api request containing in this array.
	 * @return Array
	 */
	public function wps_uwgc_giftcard_redeem( $request ) {
		$parameters    = json_decode($request->get_body(), true );
		require_once WPS_UWGC_DIRPATH . 'package/rest-api/version1/class-giftcard-for-woocommerce-rest-api-process.php';
		$wps_uwgc_rest_api_obj     = new Giftcard_For_Woocommerce_Rest_Api_Process();
		$wps_uwgc_resultsdata = $wps_uwgc_rest_api_obj->wps_uwgc_giftcard_redeem( $parameters['coupon_code'], $parameters['redeem_amount'] );
		if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 200 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata['data'], 200 );
		} else if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 404 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata, 404 ); 
		} else {
			$wps_uwgc_response = new WP_Error( $wps_uwgc_resultsdata );
		}
		return $wps_uwgc_response;
	}

	/**
	 * Returns Updated Coupon details after recharging.
	 *
	 * @param Array $request All information related with the api request containing in this array.
	 * @return Array
	 */
	public function wps_uwgc_giftcard_recharge( $request ) {
		$parameters    = json_decode($request->get_body(), true );
		require_once WPS_UWGC_DIRPATH . 'package/rest-api/version1/class-giftcard-for-woocommerce-rest-api-process.php';
		$wps_uwgc_rest_api_obj = new Giftcard_For_Woocommerce_Rest_Api_Process();
		$wps_uwgc_resultsdata = $wps_uwgc_rest_api_obj->wps_uwgc_giftcard_recharge( $parameters['coupon_code'], $parameters['recharge_amount'], $parameters['coupon_expiry'], $parameters['usage_count'], $parameters['usage_limit'] );
		if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 200 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata['data'], 200 );
		} else if ( is_array( $wps_uwgc_resultsdata ) && isset( $wps_uwgc_resultsdata['status'] ) && 404 === $wps_uwgc_resultsdata['status'] ) {
			unset( $wps_uwgc_resultsdata['status'] );
			$wps_uwgc_response = new WP_REST_Response( $wps_uwgc_resultsdata, 404 ); 
		} else {
			$wps_uwgc_response = new WP_Error( $wps_uwgc_resultsdata );
		}
		return $wps_uwgc_response;
	}

}
