<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * Installer class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class Installer
{

    private static $instance = null;

    public static function getInstance()
    {
        if (self::$instance == null)
            self::$instance = new self;

        return self::$instance;
    }

    private function __construct()
    {
        if (!empty($GLOBALS['pagenow']) && $GLOBALS['pagenow'] == 'plugins.php')
        {
            \add_action('admin_init', array($this, 'requirements'), 0);
        }

        \add_action('admin_init', array($this, 'upgrade'));
        \add_action('admin_init', array($this, 'redirect_after_activation'));
    }

    public static function activate()
    {
        if (!\current_user_can('activate_plugins'))
            return;

        self::requirements();

        if (!\wp_next_scheduled('affeggcron'))
        {
            \wp_schedule_event(time(), 'hourly', 'affeggcron');
        }
        \add_option('affegg_do_activation_redirect', true);
        \add_option('affegg_first_activation_date', time());
        self::upgrade_tables();
    }

    public static function deactivate()
    {
        \wp_clear_scheduled_hook('affeggcron');
    }

    public static function requirements()
    {
        $php_min_version = '7.4';
        $extensions = array(
            'simplexml',
            'mbstring',
        );

        $errors = array();
        $name = get_file_data(PLUGIN_FILE, array('Plugin Name'), 'plugin');

        global $wp_version;
        if (version_compare(Plugin::wp_requires, $wp_version, '>'))
            $errors[] = sprintf('You are using WordPress %s. <em>%s</em> requires at least <strong>WordPress %s</strong>.', $wp_version, $name[0], Plugin::wp_requires);
        $php_current_version = phpversion();
        if (version_compare($php_min_version, $php_current_version, '>'))
            $errors[] = sprintf('PHP is installed on your server %s. <em>%s</em> requires at least <strong>PHP %s</strong>.', $php_current_version, $name[0], $php_min_version);

        foreach ($extensions as $extension)
        {
            if (!extension_loaded($extension))
                $errors[] = sprintf('Requires extension <strong>%s</strong>.', $extension);
        }
        if (!$errors)
            return;
        unset($_GET['activate']);
        deactivate_plugins(plugin_basename(PLUGIN_FILE));
        $e = sprintf('<div class="error"><p>%1$s</p><p><em>%2$s</em> ' . 'cannot be installed!' . '</p></div>', join('</p><p>', $errors), $name[0]);
        wp_die($e);
    }

    public static function upgrade()
    {
        $settings = get_option('affegg_extractor', array());
        $affegg_db_version = (int) get_option('affegg_db_version');

        if ($affegg_db_version >= Plugin::db_version)
            return;
        self::upgrade_tables();
        if ($affegg_db_version < 37)
        {
            self::upgrade_v37();
        }
        if ($affegg_db_version < 57)
        {
            self::upgrade_v57();
        }
    }

    private static function upgrade_v37()
    {
        $upload_dir = wp_upload_dir();
        $wpdb = ProductModel::model()->getDb();
        $sql = $wpdb->prepare("UPDATE " . ProductModel::model()->tableName() . " SET img_file = REPLACE(img_file, %s, '') WHERE img_file != ''", trailingslashit($upload_dir['basedir']));
        ProductModel::model()->getDb()->query($sql);
    }

    private static function upgrade_v57()
    {
        // One-time migration:
        //  - *_domains => routing_rules
        //  - proxycrawl_domains => crawlbase routing rules
        //  - proxycrawl_token   => crawlbase_token

        $providers    = ExtractorConfig::get_scraping_providers();
        $provider_ids = array_keys($providers);

        // Read existing settings
        $settings = get_option('affegg_extractor', array());
        if (!is_array($settings))
        {
            $settings = array();
        }

        $routing_rules = array();
        if (!empty($settings['routing_rules']) && is_array($settings['routing_rules']))
        {
            $routing_rules = $settings['routing_rules'];
        }

        foreach ($provider_ids as $provider_id)
        {
            // Normal key: <provider>_domains
            $domain_keys = array($provider_id . '_domains');

            // Historical name: proxycrawl_domains should be treated as crawlbase_domains
            if ($provider_id === 'crawlbase')
            {
                $domain_keys[] = 'proxycrawl_domains';
            }

            foreach ($domain_keys as $option_key)
            {
                if (!isset($settings[$option_key]))
                {
                    continue;
                }

                $option_value = $settings[$option_key];

                if (!is_string($option_value) || $option_value === '')
                {
                    // Nothing useful here; clean up and move on
                    unset($settings[$option_key]);
                    continue;
                }

                // Convert comma-separated list into array
                $domains = TextHelper::commaListArray($option_value);
                if (empty($domains) || !is_array($domains))
                {
                    unset($settings[$option_key]);
                    continue;
                }

                foreach ($domains as $domain)
                {
                    $domain = trim($domain);
                    if ($domain === '')
                    {
                        continue;
                    }

                    $routing_rules[] = array(
                        'pattern'  => $domain,
                        'provider' => $provider_id,
                        'params'   => '',
                    );
                }

                // Remove old *_domains / proxycrawl_domains option
                unset($settings[$option_key]);
            }
        }
        if (!empty($routing_rules))
        {
            $routing_rules = ExtractorConfig::formatRoutingRules($routing_rules);
        }

        // Save new routing rules back into settings
        $settings['routing_rules'] = $routing_rules;

        // Migrate proxycrawl_token -> crawlbase_token (only if crawlbase_token not set or empty)
        $crawlbase_token  = isset($settings['crawlbase_token']) ? $settings['crawlbase_token'] : '';
        $proxycrawl_token = isset($settings['proxycrawl_token']) ? $settings['proxycrawl_token'] : '';

        if (($crawlbase_token === '' || $crawlbase_token === null) && $proxycrawl_token !== '')
        {
            $settings['crawlbase_token'] = $proxycrawl_token;
        }

        // Remove old proxycrawl_token key to avoid confusion
        if (isset($settings['proxycrawl_token']))
        {
            unset($settings['proxycrawl_token']);
        }

        update_option('affegg_extractor', $settings);
    }

    public function redirect_after_activation()
    {
        if (\get_option('affegg_do_activation_redirect', false))
        {
            delete_option('affegg_do_activation_redirect');
            wp_redirect(get_admin_url(get_current_blog_id(), 'admin.php?page=affiliate-egg'));
        }
    }

    public static function uninstall()
    {
        \delete_option('affegg_db_version');
        \delete_option(LicConfig::getInstance()->option_name());
    }

    private static function upgrade_tables()
    {
        $models = array('EggModel', 'CatalogModel', 'ProductModel', 'AutoblogModel', 'AutoblogItemModel', 'PriceHistoryModel', 'PriceAlertModel');
        $sql = '';
        foreach ($models as $model)
        {
            $m = NS . $model;
            $sql .= $m::model()->getDump();
            $sql .= "\r\n";
        }
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        update_option('affegg_db_version', Plugin::db_version);
    }
}
