<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * ExtractorConfig class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class ExtractorConfig extends Config
{

    public function page_slug()
    {
        return 'affiliate-egg-extractor-settings';
    }

    public function option_name()
    {
        return 'affegg_extractor';
    }

    public function add_admin_menu()
    {
        \add_submenu_page('options.php', __('Extractor settings', 'affegg') . ' &lsaquo; Affiliate Egg', __('Proxy settings', 'affegg'), 'manage_options', $this->page_slug, array($this, 'settings_page'));
    }

    public static function get_scraping_providers()
    {
        return array(
            'scraperapi' => array(
                'id'   => 'scraperapi',
                'name' => __('ScraperAPI', 'affegg'),
                'link' => 'https://www.keywordrush.com/go/scraperapi',
            ),
            'scrapingdog' => array(
                'id'   => 'scrapingdog',
                'name' => __('Scrapingdog', 'affegg'),
                'link' => 'https://www.keywordrush.com/go/scrapingdog',
            ),
            'scrapeowl' => array(
                'id'   => 'scrapeowl',
                'name' => __('ScrapeOwl', 'affegg'),
                'link' => 'https://scrapeowl.com/',
            ),
            'crawlbase' => array(
                'id'   => 'crawlbase',
                'name' => __('Crawlbase', 'affegg'),
                'link' => 'https://crawlbase.com/',
            ),
        );
    }

    protected function options()
    {

        $option = $this->get_scraping_provider_options();

        $options = $option + array(

            'direct_request' => array(
                'title' => __('Direct request', 'affegg'),
                'description' => __('Make the first attempt without using scrapping services.', 'affegg'),
                'callback' => array($this, 'render_dropdown'),
                'dropdown_options' => array(
                    'enabled' => __('Enabled', 'affegg'),
                    'disabled' => __('Disabled', 'affegg'),
                ),
                'default' => 'disabled',
            ),
            'retry_attempts' => array(
                'title' => __('Retry attempts', 'affegg'),
                'description' => __('The maximum number of retries when failed request. Must be between 0 to 5.', 'affegg'),
                'callback' => array($this, 'render_input'),
                'default' => '0',
                'validator' => array(
                    'trim',
                    'abs',
                    array(
                        'call' => array('\Keywordrush\AffiliateEgg\FormValidator', 'less_than_equal_to'),
                        'arg' => 5,
                        'message' => sprintf(__('The field "%s" can not be more than %d.', 'affegg'), 'Retry attempts', 5),
                    ),
                ),
            ),
        );

        return $options;
    }

    public function settings_page()
    {
        AffiliateEggAdmin::getInstance()->render('settings', array('page_slug' => $this->page_slug()));
    }

    public function proxyDomainsFilter($value)
    {
        $domains = TextHelper::commaListArray($value);
        $results = array();
        foreach ($domains as $domain)
        {
            if ($h = TextHelper::getHostName($domain))
                $host = $h;
            else
                $host = preg_replace('/^www\./', '', strtolower(trim(\sanitize_text_field($domain))));

            if ($host && TextHelper::isValidDomainName($host))
                $results[] = $host;
        }
        $results = array_unique($results);
        return join(',', $results);
    }
    public function render_routing_rules_block($args)
    {
        $rules = array();

        if (!empty($args['value']) && is_array($args['value']))
        {
            $rules = $args['value'];
        }

        // At least one empty row
        if (empty($rules))
        {
            $rules[] = array();
        }

        $total_rules = is_array($rules) ? count($rules) : 0;
        $has_many_rules = $total_rules > 15;
?>

        <div class="affegg-routing-rules-wrapper<?php echo $has_many_rules ? ' has-scroll' : ''; ?>">
            <table class="widefat striped" id="affegg-routing-rules-table">
                <thead>
                    <tr>
                        <th style="width:40%;padding-left:10px;">
                            <div class="affegg-routing-header">
                                <span><?php echo \esc_html__('Pattern', 'affegg'); ?></span>
                                <input type="search"
                                    id="affegg-routing-rules-filter"
                                    class="affegg-routing-filter-input"
                                    style="padding:2px 5px;margin-left:10px;font-weight:normal;"
                                    placeholder="<?php esc_attr_e('Type to filter…', 'affegg'); ?>">
                            </div>
                        </th>

                        <th style="width:15%;padding-left:10px;"><?php echo \esc_html__('Crawler', 'affegg'); ?></th>
                        <th style="width:40%;padding-left:10px;"><?php echo \esc_html__('Additional parameters', 'affegg'); ?></th>
                        <th style="width:5%;"></th>
                    </tr>
                </thead>

                <tbody>
                    <?php
                    foreach ($rules as $i => $rule)
                    {
                        $args['_field'] = $i;
                        $this->render_routing_rules_row($args);
                    }
                    ?>
                </tbody>
            </table>
        </div>

        <p>
            <button type="button" class="button" id="affegg-add-routing-rule">
                <?php echo \esc_html__('Add rule', 'affegg'); ?>
            </button>
        </p>

        <p class="description">
            <?php echo \esc_html__('Pattern examples:', 'affegg'); ?>
            <code>example.com</code>,
            <code>*.example.com</code>,
            <code>example.com/path/*</code><br />

            <?php echo \esc_html__('Rules are checked from top to bottom. The first matching rule will be used.', 'affegg'); ?><br />

            <?php echo \esc_html__('Additional parameters will be appended to the provider API request, for example:', 'affegg'); ?>
            <code>premium=true&amp;country_code=us</code>
        </p>

        <?php
        if (!empty($args['description']))
        {
            echo '<p class="description">' . \wp_kses_post($args['description']) . '</p>';
        }

        $this->render_routing_rules_js();
    }

    public function render_routing_rules_row($args)
    {
        $i = isset($args['_field']) ? (int) $args['_field'] : 0;

        $value = isset($args['value'][$i]) && is_array($args['value'][$i]) ? $args['value'][$i] : array();

        $pattern  = isset($value['pattern']) ? $value['pattern'] : '';
        $provider = isset($value['provider']) ? $value['provider'] : '';
        $params   = isset($value['params']) ? $value['params'] : '';

        $option_name = isset($args['option_name']) ? $args['option_name'] : '';
        $field_name  = isset($args['name']) ? $args['name'] : 'routing_rules';

        echo '<tr data-index="' . \esc_attr($i) . '">';

        // Pattern
        echo '<td>';
        echo '<input type="text" style="width:100%;max-width:100%;" class="regular-text" name="' . \esc_attr($option_name) . '['
            . \esc_attr($field_name) . '][' . $i . '][pattern]" value="'
            . \esc_attr($pattern) . '" placeholder="'
            . \esc_attr(__('Domain or pattern, e.g. example.com', 'affegg')) . '"/>';
        echo '</td>';

        // Provider
        echo '<td>';

        $providers = self::get_scraping_providers();

        echo '<select name="' . \esc_attr($option_name) . '['
            . \esc_attr($field_name) . '][' . $i . '][provider]">';
        echo '<option value="">' . \esc_html__('— Select —', 'affegg') . '</option>';

        foreach ($providers as $provider_id => $provider_data)
        {
            echo '<option value="' . \esc_attr($provider_id) . '"'
                . \selected($provider, $provider_id, false) . '>'
                . \esc_html($provider_data['name']) . '</option>';
        }

        echo '</select>';
        echo '</td>';

        // Additional params
        echo '<td>';
        echo '<input type="text" style="width:100%;max-width:100%;" class="regular-text" name="' . \esc_attr($option_name) . '['
            . \esc_attr($field_name) . '][' . $i . '][params]" value="'
            . \esc_attr($params) . '" placeholder="'
            . \esc_attr(__('Additional query string, e.g. premium=true&country_code=us', 'affegg')) . '"/>';
        echo '</td>';

        // Remove button
        echo '<td style="text-align:center;">';
        echo '<button type="button" class="button-link affegg-routing-rule-remove" title="'
            . \esc_attr__('Remove rule', 'affegg') . '">×</button>';
        echo '</td>';

        echo '</tr>';
    }

    protected function render_routing_rules_js()
    {
        static $printed = false;

        if ($printed)
        {
            return;
        }

        $printed = true;
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                var $table = $('#affegg-routing-rules-table');
                var $tbody = $table.find('tbody');

                function reindexRows() {
                    $tbody.find('tr').each(function(index) {
                        var $row = $(this);
                        $row.attr('data-index', index);

                        $row.find('input, select').each(function() {
                            var $field = $(this);
                            var name = $field.attr('name');

                            if (!name) {
                                return;
                            }

                            // Replace the numeric index in: [routing_rules][0][pattern]
                            name = name.replace(/\[\d+\]\[(pattern|provider|params)\]/, '[' + index + '][$1]');
                            $field.attr('name', name);
                        });
                    });
                }

                $('#affegg-add-routing-rule').on('click', function(e) {
                    e.preventDefault();

                    var $lastRow = $tbody.find('tr:last');
                    var $newRow;

                    if ($lastRow.length) {
                        $newRow = $lastRow.clone();

                        // Clear values in cloned row
                        $newRow.find('input').val('');
                        $newRow.find('select').prop('selectedIndex', 0);

                        $newRow.appendTo($tbody);
                        reindexRows();
                    }
                });

                $tbody.on('click', '.affegg-routing-rule-remove', function(e) {
                    e.preventDefault();

                    var $rows = $tbody.find('tr');

                    if ($rows.length <= 1) {
                        // If only one row left, just clear its values instead of removing.
                        var $row = $(this).closest('tr');
                        $row.find('input').val('');
                        $row.find('select').prop('selectedIndex', 0);
                        return;
                    }

                    $(this).closest('tr').remove();
                    reindexRows();
                });
            });
        </script>
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                var $table = $('#affegg-routing-rules-table');
                var $tbody = $table.find('tbody');

                function reindexRows() {
                    $tbody.find('tr').each(function(index) {
                        var $row = $(this);
                        $row.attr('data-index', index);

                        $row.find('input, select').each(function() {
                            var $field = $(this);
                            var name = $field.attr('name');

                            if (!name) {
                                return;
                            }

                            name = name.replace(/\[\d+\]\[(pattern|provider|params)\]/, '[' + index + '][$1]');
                            $field.attr('name', name);
                        });
                    });
                }

                $('#affegg-add-routing-rule').on('click', function(e) {
                    e.preventDefault();

                    var $lastRow = $tbody.find('tr:last');
                    var $newRow;

                    if ($lastRow.length) {
                        $newRow = $lastRow.clone();

                        $newRow.find('input').val('');
                        $newRow.find('select').prop('selectedIndex', 0);

                        $newRow.appendTo($tbody);
                        reindexRows();
                    }
                });

                $tbody.on('click', '.affegg-routing-rule-remove', function(e) {
                    e.preventDefault();

                    var $rows = $tbody.find('tr');

                    if ($rows.length <= 1) {
                        var $row = $(this).closest('tr');
                        $row.find('input').val('');
                        $row.find('select').prop('selectedIndex', 0);
                        return;
                    }

                    $(this).closest('tr').remove();
                    reindexRows();
                });

                // Simple filter by pattern
                var $filter = $('#affegg-routing-rules-filter');

                $filter.on('keyup change', function() {
                    var term = $.trim($(this).val()).toLowerCase();

                    $tbody.find('tr').each(function() {
                        var $row = $(this);

                        // Look at the Pattern input specifically:
                        var patternVal = $row.find('input[name*="[pattern]"]').val() || '';

                        if (term === '') {
                            $row.show();
                        } else if (patternVal.toLowerCase().indexOf(term) !== -1) {
                            $row.show();
                        } else {
                            $row.hide();
                        }
                    });
                });
            });
        </script>
<?php
    }

    public static function formatRoutingRules($rules)
    {
        if (!is_array($rules))
        {
            return array();
        }

        $providers         = self::get_scraping_providers();
        $allowed_providers = array_keys($providers);
        $clean             = array();

        foreach ($rules as $rule)
        {
            if (!is_array($rule))
            {
                continue;
            }

            $pattern_raw  = isset($rule['pattern'])  ? wp_unslash($rule['pattern'])  : '';
            $provider_raw = isset($rule['provider']) ? wp_unslash($rule['provider']) : '';
            $params_raw   = isset($rule['params'])   ? wp_unslash($rule['params'])   : '';

            // Normalize pattern

            $pattern = trim($pattern_raw);

            if ($pattern !== '')
            {
                // If user pasted a full URL (has scheme), extract host + path
                if (preg_match('~^[a-z][a-z0-9+\-.]*://~i', $pattern))
                {
                    $parsed = wp_parse_url($pattern);

                    if (!empty($parsed['host']))
                    {
                        $host = strtolower($parsed['host']); // normalize host
                        $path = isset($parsed['path']) ? $parsed['path'] : '';

                        // Build pattern as "host" or "host/path..."
                        $pattern = $host . $path;
                    }
                }

                // Final sanitization (keeps *, ., /, etc. but strips weird stuff)
                $pattern = sanitize_text_field($pattern);
            }

            $provider = $provider_raw !== '' ? sanitize_key($provider_raw) : '';
            $params = $params_raw !== '' ? sanitize_text_field($params_raw) : '';

            // Skip rows without required fields
            if ($pattern === '' || $provider === '')
            {
                continue;
            }

            if (!in_array($provider, $allowed_providers, true))
            {
                continue;
            }

            $clean[] = array(
                'pattern'  => $pattern,
                'provider' => $provider,
                'params'   => $params,
            );
        }

        return array_values($clean);
    }

    protected function get_scraping_provider_options()
    {
        $options = array();

        foreach (self::get_scraping_providers() as $provider_id => $provider)
        {
            $option_key = $provider_id . '_token';

            $title = sprintf(
                /* translators: %s: provider name */
                __('%s API key', 'affegg'),
                $provider['name']
            );

            if (!empty($provider['link']))
            {
                $description = sprintf(
                    /* translators: 1: provider URL, 2: provider name */
                    __('Your <a target="_blank" href="%1$s">%2$s</a> token.', 'affegg'),
                    esc_url($provider['link']),
                    esc_html(wp_strip_all_tags($provider['name']))
                );
            }
            else
            {
                $description = sprintf(
                    /* translators: %s: provider name */
                    __('Your %s token.', 'affegg'),
                    esc_html(wp_strip_all_tags($provider['name']))
                );
            }

            $options[$option_key] = array(
                'title'       => $title,
                'description' => $description,
                'callback'    => array($this, 'render_password'),
                'default'     => '',
                'validator'   => array(
                    'trim',
                ),
            );
        }

        $options['routing_rules'] = array(
            'title'       => __('Routing rules', 'affegg'),
            'description' => __(
                'Define which scraping provider should be used for particular domains or URL patterns. Rules are processed from top to bottom; the first match wins.',
                'affegg'
            ),
            'callback'  => array($this, 'render_routing_rules_block'),
            'default'   => array(),
            'validator' => array(
                array(
                    'call' => array(self::class, 'formatRoutingRules'),
                    'type' => 'filter',
                ),
            ),
        );

        return $options;
    }
}
