<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * AmazoncomParser class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class AmazoncomParser extends ShopParser
{
    protected $canonical_domain = 'https://www.amazon.com';
    protected $charset = 'utf-8';
    protected $currency = 'USD';
    protected $user_agent = array('Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:138.0) Gecko/20100101 Firefox/138.0');
    protected $headers = array(
        'Accept' => 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
        'Accept-Language' => 'en-us,en;q=0.5',
        'Cache-Control' => 'no-cache',
    );
    protected $_html;

    public function restPostGet($url, $fix_encoding = true)
    {
        \add_action('http_api_curl', array(__CLASS__, '_setCurlOptions'), 10, 3);

        $body = parent::restPostGet($url, false);

        // fix html
        $body = preg_replace('/<table id="HLCXComparisonTable".+?<\/table>/uims', '', $body);
        $b = preg_replace('/<head\b[^>]*>(.*?)<\/head>/uims', '', $body);

        if ($b)
            $body = $body;
        $body = preg_replace('/<script.*?>.*?<\/script>/uims', '', $body);
        $body = preg_replace('/<style.*?>.*?<\/style>/uims', '', $body);

        $this->_html = $this->decodeCharset($body, $fix_encoding);
        return $this->_html;
    }

    static public function _setCurlOptions($handle, $r, $url)
    {
        curl_setopt($handle, CURLOPT_ENCODING, '');
    }

    public function parseCatalog($max)
    {
        $xpath = array(
            ".//div[contains(@class, 'SEARCH_RESULTS')]//a[contains(@class, 'a-link-normal') and contains(@href, '/dp/')]/@href",
            ".//div[@class='p13n-desktop-grid']//a[@class='a-link-normal']/@href",
            ".//*[@class='aok-inline-block zg-item']/a[@class='a-link-normal']/@href",
            ".//h3[@class='newaps']/a/@href",
            ".//div[@id='resultsCol']//a[contains(@class,'s-access-detail-page')]/@href",
            ".//div[@class='zg_title']/a/@href",
            ".//div[@id='rightResultsATF']//a[contains(@class,'s-access-detail-page')]/@href",
            ".//div[@id='atfResults']/ul//li//div[contains(@class,'a-column')]/a/@href",
            ".//div[@id='mainResults']//li//a[@title]/@href",
            ".//*[@id='zg_centerListWrapper']//a[@class='a-link-normal' and not(@title)]/@href",
            ".//h5/a[@class='a-link-normal a-text-normal']/@href",
            ".//*[@data-component-type='s-product-image']//a[@class='a-link-normal']/@href",
            ".//div[@class='a-section a-spacing-none']/h2/a/@href",
            ".//h2/a[@class='a-link-normal a-text-normal']/@href",
            ".//span[@data-component-type='s-product-image']/a/@href",
        );

        $urls = $this->xpathArray($xpath);

        // Today's Deals
        if (!$urls)
            $urls = $this->parseGoldBoxDeals();

        if (!$urls)
            return array();

        foreach ($urls as $i => $url)
        {
            if (!preg_match('/^https?:\/\//', $url))
                $urls[$i] = $this->canonical_domain . $url;
        }

        // picassoRedirect fix
        foreach ($urls as $i => $url)
        {
            if (!strstr($url, '/gp/slredirect/picassoRedirect.html/'))
                continue;
            $parts = parse_url($url);
            if (empty($parts['query']))
                continue;
            parse_str($parts['query'], $output);
            if (isset($output['url']))
                $urls[$i] = $output['url'];
            else
                unset($urls[$i]);
        }

        // fix urls. prevent duplicates for autobloging
        $res = array();
        foreach ($urls as $key => $url)
        {
            if ($asin = self::parseAsinFromUrl($url))
                $res[] = $this->canonical_domain . '/dp/' . $asin . '/';
        }

        $res = array_unique($res);
        $res = array_values($res);

        return $res;
    }

    protected function parseGoldBoxDeals()
    {

        if (!strstr($this->getUrl(), 'amazon.com/gp/goldbox'))
            return array();

        $request_url = 'https://www.amazon.com/xa/dealcontent/v2/GetDeals?nocache=' . time();
        $response = \wp_remote_post($request_url, array(
            'headers' => array('Content-Type' => 'application/json; charset=utf-8'),
            'body' => '{"requestMetadata":{"marketplaceID":"ATVPDKIKX0DER","clientID":"goldbox_mobile_pc","sessionID":"147-0111701-3832735"},"dealTargets":[{"dealID":"27929040"},{"dealID":"2dfcb07b"},{"dealID":"6727cdb5"},{"dealID":"676b0c2d"},{"dealID":"7aeb0274"},{"dealID":"7ca1692e"},{"dealID":"a6614039"},{"dealID":"af1e3631"},{"dealID":"b3db4ae7"},{"dealID":"e2b741c7"},{"dealID":"eb7ca674"},{"dealID":"f5a1f5c0"}],"responseSize":"ALL","itemResponseSize":"DEFAULT_WITH_PREEMPTIVE_LEAKING","widgetContext":{"pageType":"Landing","subPageType":"hybrid-batch-btf","deviceType":"pc","refRID":"KH2KVAGJESZ5EF3NCNGD","widgetID":"f3f63185-46c5-4297-bc5f-35921b3fb364","slotName":"merchandised-search-8"},"customerContext":{"languageOfPreference":"en_US"}}',
            'method' => 'POST'
        ));

        if (\is_wp_error($response) || !$body = \wp_remote_retrieve_body($response))
            return array();
        $js_data = json_decode($body, true);

        if (!$js_data || !isset($js_data['dealDetails']))
            return array();

        $urls = array();
        foreach ($js_data['dealDetails'] as $hit)
        {
            if (strstr($hit['egressUrl'], '/dp/'))
                $urls[] = $hit['egressUrl'];
        }
        return $urls;
    }

    public function parseTitle()
    {
        $paths = array(
            ".//h1[@id='title']/span",
            ".//*[@id='fine-ART-ProductLabelArtistNameLink']",
            ".//meta[@name='title']/@content",
            ".//h1",
        );

        return $this->xpathScalar($paths);
    }

    public function parseDescription()
    {
        $path = array(
            ".//div[@id='featurebullets_feature_div']//span[@class='a-list-item']",
            ".//div[@id='featurebullets_feature_div']//li",
            ".//h3[@class='product-facts-title']//..//li",
        );

        if ($results = $this->xpathArray($path))
        {
            $results = array_map('\sanitize_text_field', $results);
            $key = array_search('Make sure this fits by entering your model number.', $results);
            if ($key !== false)
                unset($results[$key]);
            return '<ul><li>' . implode("</li><li>\n", $results) . '</li></ul>';
        }

        $result = $this->xpathScalar(".//script[contains(.,'iframeContent')]");
        if ($result && preg_match('/iframeContent\s=\s"(.+?)"/msi', $result, $match))
        {
            $res = urldecode($match[1]);
            if (preg_match('/class="productDescriptionWrapper">(.+?)</msi', $res, $match))
                return trim($match[1]);
        }

        $paths = array(
            ".//*[@id='bookDescription_feature_div']/noscript/div",
            ".//*[@id='productDescription']//*[@class='productDescriptionWrapper']",
            ".//*[@id='productDescription']/p/*[@class='btext']",
            ".//*[@id='bookDescription_feature_div']/noscript",
            ".//*[@class='dv-simple-synopsis dv-extender']",
            ".//*[@id='bookDescription_feature_div']//noscript/div",
            ".//div[@id='bookDescription_feature_div']",
            ".//*[@id='productDescription']/p",

        );
        if ($description = $this->xpathScalar($paths, true))
            return $description;

        if (preg_match('/bookDescEncodedData = "(.+?)",/', $this->_html, $matches))
            return html_entity_decode(urldecode($matches[1]));

        if (preg_match('/(<div id="bookDescription_feature_div".+?)<a href="/ims', $this->_html, $matches))
            return $matches[1];

        return '';
    }

    public function parsePrice()
    {
        if (!$this->isInStock())
            return;

        $price = $this->xpathScalar(".//span[@id='style_name_0_price']/span[contains(@class, 'olpWrapper')]");
        if ($price && strstr($price, ' options from '))
        {
            $parts = explode('options from', $price);
            return $parts[1];
        }

        $paths = array(
            ".//span[@class='a-price aok-align-center reinventPricePriceToPayMargin priceToPay']",
            ".//span[@id='subscriptionPrice']//span[@data-a-color='price']//span[@class='a-offscreen']",
            ".//table[@class='a-lineitem a-align-top']//span[@data-a-color='price']//span[@class='a-offscreen']",
            ".//*[contains(@class, 'priceToPay')]//*[@class='a-offscreen']",
            ".//*[@class='a-price aok-align-center reinventPricePriceToPayMargin priceToPay']",
            ".//div[@class='a-section a-spacing-none aok-align-center']//span[@class='a-offscreen']",
            ".//span[contains(@class, 'a-price') and contains(@class, 'priceToPay')]//span[@class='a-offscreen']",
            ".//h5//span[@id='price']",
            ".//span[@class='a-price a-text-price header-price a-size-base a-text-normal']//span[@class='a-offscreen']",
            ".//span[@class='a-price a-text-price a-size-medium apexPriceToPay']//span[@class='a-offscreen']",
            ".//span[contains(@class, 'priceToPay')]",
            ".//div[@class='a-section a-spacing-small a-spacing-top-small']//a/span[@class='a-size-base a-color-price']",
            ".//div[@class='a-section a-spacing-none aok-align-center']//span[@class='a-offscreen']",
            ".//*[@id='priceblock_dealprice']",
            ".//span[@id='priceblock_ourprice']",
            ".//span[@id='priceblock_saleprice']",
            ".//div[@class='twisterSlotDiv addTwisterPadding']//span[@id='color_name_0_price']",
            ".//input[@name='displayedPrice']/@value",
            ".//*[@id='unqualifiedBuyBox']//*[@class='a-color-price']",
            ".//*[@class='dv-button-text']",
            ".//*[@id='cerberus-data-metrics']/@data-asin-price",
            ".//div[@id='olp-upd-new-freeshipping']//span[@class='a-color-price']",
            ".//span[@id='rentPrice']",
            ".//span[@id='newBuyBoxPrice']",
            ".//div[@id='olp-new']//span[@class='a-size-base a-color-price']",
            ".//span[@id='unqualified-buybox-olp']//span[@class='a-color-price']",
            ".//span[@id='price_inside_buybox']",
            ".//span[@class='slot-price']//span[@class='a-size-base a-color-price a-color-price']",
            ".//span[@class='a-button-inner']//span[contains(@class, 'a-color-price')]",
            ".//div[@id='booksHeaderSection']//span[@id='price']",
            ".//div[@class='a-box-inner a-padding-base']//span[@class='a-color-price aok-nowrap']",
            ".//span[@id='kindle-price']",
            ".//span[contains(@class, 'a-price')]//span/@aria-hidden",
            ".//span[contains(@class, 'a-price')]//span[@class='a-price-whole']",
        );

        $price = $this->xpathScalar($paths);
        $price = str_replace('R$', '', $price);

        if (!$price && $price = $this->xpathScalar(".//span[@id='priceblock_ourprice']//*[@class='buyingPrice' or @class='price-large']"))
        {
            if ($cent = $this->xpathScalar(".//span[@id='priceblock_ourprice']//*[@class='verticalAlign a-size-large priceToPayPadding' or @class='a-size-small price-info-superscript']"))
                $price = $price . '.' . $cent;
        }

        if (strstr($price, ' - '))
        {
            $tprice = explode('-', $price);
            $price = $tprice[0];
        }

        return $price;
    }

    public function parseOldPrice()
    {
        if (!$this->isInStock())
            return;

        $paths = array(
            ".//*[not(@class='pricePerUnit')]//span[@class='a-price a-text-price a-size-base']//span[@class='a-offscreen']",
            ".//*[@id='price']//span[@class='a-text-strike']",
            ".//div[@id='price']//td[contains(@class,'a-text-strike')]",
            "(.//*[@id='price']//span[@class='a-text-strike'])[2]",
            ".//*[@id='buyBoxInner']//*[contains(@class, 'a-text-strike')]",
            ".//*[@id='price']//span[contains(@class, 'priceBlockStrikePriceString')]",
            ".//span[@id='rentListPrice']",
            ".//span[@id='listPrice']",
            ".//*[not(@class='pricePerUnit')]//span[@class='a-price a-text-price a-size-base']/span[@class='a-offscreen']",
            ".//span[@class='a-size-small a-color-secondary aok-align-center basisPrice']//span[@class='a-price a-text-price']/span[@class='a-offscreen']",

        );

        return $this->xpathScalar($paths);
    }

    public function parseManufacturer()
    {
        $brand = $this->xpathScalar(".//a[@id='brand']");
        if (!$brand)
            $brand = $this->xpathScalar(".//*[@id='byline']//*[contains(@class, 'contributorNameID')]");
        return $brand;
    }

    public function parseImg()
    {
        $paths = array(
            ".//img[@id='miniATF_image']/@src",
            ".//img[@id='landingImage']/@data-old-hires",
            ".//img[@id='landingImage']/@data-a-dynamic-image",
            ".//img[@id='landingImage']/@src",
            ".//img[@id='ebooksImgBlkFront']/@src",
            ".//*[@id='fine-art-landingImage']/@src",
            ".//*[@class='dv-dp-packshot js-hide-on-play-left']//img/@src",
            ".//*[@id='main-image']/@src",
            ".//div[@id='mainImageContainer']/img/@src",
            ".//img[@id='imgBlkFront' and not(contains(@src, 'data:image'))]/@src",
            ".//div[@id='imgTagWrapperId']//img/@src",
        );

        $img = $this->xpathScalar($paths);

        if (preg_match('/^data:image/', $img))
            $img = '';

        if (preg_match('/"(https:\/\/.+?)"/', $img, $matches))
            $img = $matches[1];

        if (!$img)
        {
            $dynamic = $this->xpathScalar(".//img[@id='landingImage' or @id='imgBlkFront']/@data-a-dynamic-image");
            if (preg_match('/"(https:\/\/.+?)"/', $dynamic, $matches))
                $img = $matches[1];
        }
        if (!$img)
        {
            $img = $this->xpathScalar(".//img[@id='imgBlkFront']/@src");
            if (preg_match('/^data:image/', $img))
                $img = '';
        }

        if (!$img)
        {
            $img = $this->xpathScalar(".//*[contains(@class, 'imageThumb thumb')]/img/@src");
            $img = preg_replace('/\._.+?\_.jpg/', '.jpg', $img);
        }

        $img = str_replace('._SL1500_.', '._AC_SL520_.', $img);
        $img = str_replace('._SL1200_.', '._AC_SL520_.', $img);
        $img = str_replace('._SL1000_.', '._AC_SL520_.', $img);
        $img = str_replace('._AC_SL1500_.', '._AC_SL520_.', $img);
        $img = str_replace('._AC_UL1192_.', '._AC_SL520_.', $img);

        return $img;
    }

    public function parseImgLarge()
    {
        $img = $this->parseImg();
        $img = str_replace('._AC_SL520_.', '._SL1500_.', $img);
        return $img;
    }

    public function parseExtra()
    {
        $extra = parent::parseExtra();

        $extra['categoryPath'] = $this->xpathArray(".//div[@id='wayfinding-breadcrumbs_feature_div']//li//a");;

        $extra['comments'] = array();
        $review_titles = array();
        $comments = $this->xpathArray(".//*[contains(@class, 'reviews-content')]//*[contains(@data-hook, 'review-body')]//div[@data-hook]", true);
        if ($comments)
        {
            $users = $this->xpathArray(".//*[contains(@class, 'reviews-content')]//*[@class='a-profile-name']");
            $dates = $this->xpathArray(".//*[contains(@class, 'reviews-content')]//*[@data-hook='review-date']");
            $ratings = $this->xpathArray(".//*[contains(@class, 'reviews-content')]//*[@data-hook='review-star-rating' or @data-hook='cmps-review-star-rating']");
            $review_titles = $this->xpathArray(".//*[contains(@class, 'reviews-content')]//a[@data-hook='review-title']//span[2]");
        }
        else
        {
            $comments = $this->xpathArray(".//*[@id='revMH']//*[contains(@id, 'revData-dpReviewsMostHelpful')]/div[@class='a-section']", true);
            $users = $this->xpathArray(".//*[@id='revMH']//a[@class='noTextDecoration']");
            $dates = $this->xpathArray(".//*[@id='revMH']//span[@class='a-color-secondary']/span[@class='a-color-secondary']");
            $ratings = $this->xpathArray(".//*[@id='revMH']//span[@class='a-icon-alt']");
        }

        /*
        if (!$comments)
        {
            $comments = $this->xpathArray(".//div[@id='cm-cr-dp-review-list']//div[contains(@class, 'reviewText')]", true);
            $users = $this->xpathArray(".//div[@id='cm-cr-dp-review-list']//span[@class='a-profile-name']");
            $dates = $this->xpathArray(".//div[@id='cm-cr-dp-review-list']//div[contains(@class, 'review-date')]");
            $ratings = $this->xpathArray(".//div[@id='cm-cr-dp-review-list']//i[contains(@class, 'a-icon-star')]/span");
        }
        */

        for ($i = 0; $i < count($comments); $i++)
        {
            if (isset($users[$i]))
                $comment['name'] = sanitize_text_field($users[$i]);

            $date = $dates[$i];
            if (preg_match('/Reviewed in .+? on (.+)/', $date, $matches))
                $date = $matches[1];
            elseif (preg_match('/(\d.+)/', $date, $matches))
                $date = $matches[1];
            $comment['date'] = strtotime($date);

            $comment['comment'] = '';
            if (isset($review_titles[$i]))
            {
                $review_title = sanitize_text_field($review_titles[$i]);
                $review_title = trim($review_title, ',.!? ');
                $review_title = $review_title . '.';
                $comment['comment'] .= $review_title . ' ';
            }

            $comment['comment'] .= sanitize_text_field($comments[$i]);
            $comment['comment'] = preg_replace('/Read\smore$/', '', $comment['comment']);
            if (isset($ratings[$i]))
                $comment['rating'] = $this->prepareRating($ratings[$i]);
            $extra['comments'][] = $comment;
        }

        preg_match("/\/dp\/(.+?)\//msi", $this->getUrl(), $match);
        $extra['item_id'] = isset($match[1]) ? $match[1] : '';

        $extra['images'] = $this->_parseImages();

        $extra['ratingDecimal'] = $this->_parseRatingValue();
        $extra['rating'] = $this->prepareRating($extra['ratingDecimal']);
        $extra['ratingCount'] = $this->_parseReviewCount();
        $extra['reviewUrl'] = '';

        if ($asin = self::parseAsinFromUrl($this->getUrl()))
        {
            $url_parts = parse_url($this->getUrl());
            $extra['reviewUrl'] = $url_parts['scheme'] . '://' . $url_parts['host'] . '/product-reviews/' . $asin . '/';
        }

        if ($description = $this->xpathScalar(".//div[@id='productDescription']//p"))
            $extra['product_description'] = $description;

        return $extra;
    }

    protected function _parseReviewCount()
    {
        $paths = array(
            ".//*[@id='acrCustomerReviewText']",
        );

        return (int) str_replace(',', '', $this->xpathScalar($paths));
    }

    protected function _parseRatingValue()
    {
        $paths = array(
            ".//*[@id='summaryStars']//i/span",
            ".//*[@id='acrPopover']//i[contains(@class, 'a-icon a-icon-star')]",
        );

        return (float) $this->xpathScalar($paths);
    }

    protected function _parseImages()
    {
        $images = array();
        $results = $this->xpathArray(".//div[@id='altImages']//ul/li[position() > 1]//img[contains(@src, '.jpg') and not(contains(@src, 'play-icon-overlay'))]/@src");

        foreach ($results as $img)
        {
            if (strstr($img, 'play-button'))
                continue;

            $img = preg_replace('/,\d+_\.jpg/', '.jpg', $img);
            $img = preg_replace('/\._.+?_\.jpg/', '.jpg', $img);
            $img = preg_replace('/\._SX\d+_SY\d+_.+?\.jpg/', '.jpg', $img);

            if (preg_match('/^(https:\/\/m\.media-amazon\.com\/images\/I\/[^._]+)\.\S+\.jpg$/', $img, $matches))
                $img = $matches[1] . ".jpg";

            $images[] = $img;
        }
        return $images;
    }

    public function getFeaturesXpath()
    {

        return array(
            array(
                'name' => ".//div[@id='prodDetails']//table//th",
                'value' => ".//div[@id='prodDetails']//table//td",
            ),
            array(
                'name' => ".//div[@id='productOverview_feature_div']//table//td[1]",
                'value' => ".//div[@id='productOverview_feature_div']//table//td[2]",
            ),
            array(
                'name' => ".//table[contains(@id, 'productDetails_techSpec_section')]//th",
                'value' => ".//table[contains(@id, 'productDetails_techSpec_section')]//td",
            ),
            array(
                'name' => ".//table[contains(@id, 'technicalSpecifications_section')]//th",
                'value' => ".//table[contains(@id, 'technicalSpecifications_section')]//td",
            ),
            array(
                'name' => ".//table[contains(@id, 'productDetails_detailBullets_sections')]//th",
                'value' => ".//table[contains(@id, 'productDetails_detailBullets_sections')]//td",
            ),
            array(
                'name-value' => ".//*[@id='productDetailsTable']//li[not(@id) and not(@class)]",
                'separator' => ":",
            ),
            array(
                'name' => ".//*[@id='prodDetails']//td[@class='label']",
                'value' => ".//*[@id='prodDetails']//td[@class='value']",
            ),
            array(
                'name' => ".//*[contains(@id, 'technicalSpecifications_section')]//th",
                'value' => ".//*[contains(@id, 'technicalSpecifications_section')]//td",
            ),
            array(
                'name-value' => ".//div[@id='technical-data']//li",
                'separator' => ":",
            ),
            array(
                'name-value' => ".//div[@id='detail-bullets']//li",
                'separator' => ":",
            ),
            array(
                'name' => ".//div[@id='detailBullets_feature_div']//li/span/span[1]",
                'value' => ".//div[@id='detailBullets_feature_div']//li/span/span[2]",
            ),
            array(
                'name' => ".//div[@id='tech']//table//td[1]",
                'value' => ".//div[@id='tech']//table//td[2]",
            ),
        );
    }

    public function isInStock()
    {
        if ($this->xpathScalar(".//div[@id='availability']/span[contains(@class,'a-color-success')]"))
            return true;

        $availability = trim((string)$this->xpathScalar(".//div[@id='availability']/span/text()"));

        if ($availability == 'Currently unavailable.' || $availability == 'Şu anda mevcut değil.' || $availability == 'Attualmente non disponibile.' || $availability == 'Momenteel niet verkrijgbaar.' || $availability == 'Non disponibile.')
            return false;

        return true;
    }

    private function prepareRating($rating_str)
    {
        $rating_parts = explode(' ', (string)$rating_str);
        return TextHelper::ratingPrepare($rating_parts[0]);
    }

    public static function parseAsinFromUrl($url)
    {
        $regex = '~/(?:exec/obidos/ASIN/|o/|gp/product/|gp/offer-listing/|(?:(?:[^"\'/]*)/)?dp/|)([0-9A-Z]{10})(?:(?:/|\?|\#)(?:[^"\'\s]*))?~isx';
        if (preg_match($regex, $url, $matches))
            return $matches[1];
        else
            return false;
    }

    public function getCurrency()
    {
        if (strstr($this->parsePrice(), 'USD'))
            return 'USD';

        if (strstr($this->parsePrice(), 'AUD'))
            return 'AUD';

        if (strstr($this->parsePrice(), 'DKK'))
            return 'DKK';

        if (strstr($this->parsePrice(), 'ILS'))
            return 'ILS';

        return $this->currency;
    }
}
