<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * BolcomParser class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class BolcomParser extends LdShopParser
{
    protected $charset = 'utf-8';
    protected $currency = 'EUR';

    public function parseCatalog($max)
    {
        $sponsored = $this->xpathArray(".//*[@class='dsa__label']/../../..//a[contains(@class, 'product-title')]/@href");

        $path = array(
            ".//a[@data-test='product-title']/@href",
            ".//a[@class='product-title px_list_page_product_click list_page_product_tracking_target']/@href",
        );

        $links = $this->xpathArray($path);

        return array_diff($links, $sponsored);
    }

    public function parseTitle()
    {
        $p = parent::parseTitle();

        $path = array(
            ".//span[@data-test='title']",
            ".//h1",
        );

        if ($t = $this->xpathScalar($path))
            return $t;
        else
            return $p;
    }

    public function parsePrice()
    {
        $path = array(
            ".//span[@data-test='price']",
        );

        if ($p = $this->xpathScalar($path))
        {
            $p = sanitize_text_field($p);
            $p = str_replace(' ', '.', $p);
            return $p;
        }

        if ($price = parent::parsePrice())
            return $price;

        if (isset($this->ld_json['workExample']['potentialAction']['expectsAcceptanceOf']['price']))
            return $this->ld_json['workExample']['potentialAction']['expectsAcceptanceOf']['price'];
    }

    public function parseOldPrice()
    {
        $paths = array(
            ".//del[@class='h-nowrap buy-block__list-price']",
            ".//section[contains(@class, 'buy-block__prices')]//del[@class='buy-block__list-price']",
        );

        return $this->xpathScalar($paths);
    }

    public function parseDescription()
    {
        $paths = array(
            ".//div[@class='product-description']",
        );

        if ($d = $this->xpathScalar($paths, true))
            return $d;
        else
            return parent::parseDescription();
    }

    public function parseImg()
    {
        if ($images = $this->_parseImages())
            return reset($images);

        $paths = array(
            ".//meta[@property='og:image']/@content",
        );

        return $this->xpathScalar($paths);
    }

    public function parseImgLarge()
    {
        if (preg_match('/"zoomImageUrl":"(.+?)"/', $this->dom->saveHtml(), $matches))
            return $matches[1];
    }

    public function getFeaturesXpath()
    {
        return array(
            array(
                'name' => ".//div[@class='specs']//dt/text()[normalize-space()]",
                'value' => ".//div[@class='specs']//dd",
            ),
        );
    }

    public function parseExtra()
    {
        $extra = parent::parseExtra();

        $extra['features'] = array();
        if ($features = $this->parseFeatures())
            $extra['features'] = $features;

        if ($r = $this->xpathScalar(".//div[@data-test='rating-suffix']"))
        {
            $r = explode('(', $r);
            if (count($r) == 2)
                $extra['ratingCount'] = (int) $r[1];
        }

        $extra['images'] = $this->_parseImages();
        $extra['ratingDecimal'] = $this->_parseRatingValue();
        $extra['rating'] = $extra['ratingDecimal'];

        if ($reviews = $this->parseReviews())
            $extra['comments'] = $reviews;

        return $extra;
    }

    protected function _parseImages()
    {
        $html = $this->dom->saveHtml();

        if (preg_match_all('/"zoomImageUrl":"(https?:\/\/[^"]+)"/', $html, $matches))
        {
            foreach ($matches[1] as $i => $url)
            {
                $matches[1][$i] = json_decode('"' . $url . '"');
            }

            return $matches[1];
        }

        if (preg_match_all('/"imageUrl":"(.+?)"/ims', $html, $matches))
            return $matches[1];
    }

    public function getReviewsXpath()
    {
        return array(
            array(
                'review' => ".//div[@class='reviews']//p[@data-test='review-body']",
                'rating' => ".//*[@class='reviews']//*[@class='star-rating']/span/@style",
                'author' => ".//*[@class='reviews']//ul[@class='review-metadata__list']/li[1]",
                'date' => ".//*[@class='reviews']//ul[@class='review-metadata__list']/li[3]",
            ),
        );
    }

    public function parseReviews()
    {
        $pros_cons_xpath = ".//div[@class='review-pros-and-cons']";

        if (!$xpaths = $this->getReviewsXpath())
            return array();

        if (isset($xpaths['review']))
            $xpaths = array($xpaths);

        foreach ($xpaths as $xpath)
        {
            $reviews = $ratings = $authors = $dates = array();

            if (!empty($xpath['review']))
                $reviews = $this->xpathArray($xpath['review'], true);

            if (!$reviews)
                continue;

            $pros_cons = $this->xpathArray($pros_cons_xpath, true);

            if (!empty($xpath['rating']))
                $ratings = $this->xpathArray($xpath['rating']);

            if (!empty($xpath['author']))
                $authors = $this->xpathArray($xpath['author']);

            if (!empty($xpath['date']))
                $dates = $this->xpathArray($xpath['date']);

            $results = array();
            for ($i = 0; $i < count($reviews); $i++)
            {
                $review = array();
                $review['comment'] = \normalize_whitespace(TextHelper::sanitizeHtml(html_entity_decode($reviews[$i])));

                if (isset($pros_cons[$i]))
                    $review['comment'] .= "\n" . \normalize_whitespace(TextHelper::sanitizeHtml(html_entity_decode($pros_cons[$i])));

                if (isset($ratings[$i]))
                {
                    if (strstr($ratings[$i], ':'))
                    {
                        $r_parts = explode(":", $ratings[$i]);
                        if (count($r_parts) == 2)
                            $ratings[$i] = $r_parts[1];
                    }

                    $review['rating'] = TextHelper::ratingPrepare($ratings[$i]);
                }

                if (isset($authors[$i]))
                    $review['author'] = \sanitize_text_field(html_entity_decode($authors[$i]));

                if (isset($dates[$i]))
                    $review['date'] = strtotime($dates[$i]);

                $results[] = $review;
            }

            if ($results)
                return $results;
        }
        return array();
    }

    public function isInStock()
    {
        if ($this->xpathScalar(".//div[contains(@class, 'buy-block__options')]//a[@data-button-type='buy']"))
            return true;
        else
            return parent::isInStock();
    }

    protected function _parseRatingValue()
    {
        $paths = array(
            ".//div[@data-test='rating-suffix']",
        );

        if (!$rating_str = (string) $this->xpathScalar($paths))
            return 0;

        $parts = explode('/', $rating_str);
        $rating = (float) str_replace(',', '.', $parts[0]);

        return $rating;
    }
}
