<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * BookingParser class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class BookingParser extends LdShopParser
{
    protected $charset = 'utf-8';
    protected $currency = 'USD';

    public function parseCatalog($max)
    {

        $xpath = array(
            ".//a[@data-testid='title-link']/@href",
            ".//*[@id='hotellist_inner']//h3/a/@href",
            ".//div[@class='sr__card_photo']/a/@href",
        );

        $urls = $this->xpathArray($xpath);
        foreach ($urls as $i => $url)
        {
            $url = strtok($url, '?');
            $url =  preg_replace('/\.[a-z]{2}\.html/', '.html', $url);
            $urls[$i] = $url;
        }

        return $urls;
    }

    public function parseTitle()
    {
        $ptitle = parent::parseTitle();
        $title = $this->xpathScalar(".//title");
        $parts = explode(' – ', $title);
        $title = reset($parts);
        $title = preg_replace('/\(updated prices \d+\)/i', '', $title);

        if ($title)
            return $title;
        else
            return $ptitle;
    }

    public function parseDescription()
    {
        $path = array(
            ".//*[@data-testid='property-description']",
        );

        if ($d = $this->xpathScalar($path, true))
            return $d;

        return html_entity_decode(parent::parseDescription());
    }

    public function parsePrice()
    {
        $price = (string) parent::parsePrice();

        if (preg_match('/[A-Z]{3}.+?([0-9\.\s\'\,]+)/', $price, $matches))
            $price = trim($matches[1]);
        elseif (preg_match('/\d[0-9\.,\s]+/', $price, $matches))
            $price = trim($matches[0]);

        $price = str_replace(' ', '', $price);

        return $price;
    }

    public function parseImg()
    {
        if ($images = $this->_parseImages())
            return reset($images);

        if ($img = parent::parseImg())
        {
            $img = preg_replace('~/max\d+/~', '/max1024x768/', $img);
            return $img;
        }
    }

    public function parseExtra()
    {
        $html = $this->dom->saveHtml();
        $extra = parent::parseExtra();

        $images = $this->_parseImages();

        array_shift($images);
        $extra['images'] = $images;
        if (preg_match('/"ratingValue" : (.+?),/', $html, $matches))
        {
            $rating = (float) str_replace(',', '.', $matches[1]);
            $extra['rating'] = TextHelper::ratingPrepare(TextHelper::convertRatingScale($rating, 1, 10, 1, 5));
            $extra['ratingDecimal'] = round(TextHelper::convertRatingScale($rating, 1, 10, 1, 5), 2);
            $extra['ratingScale10'] = round($rating, 2);
        }

        //Booking.com rated the quality of this property as 3 out of 5 based
        if (preg_match('/"reviewCount" : (\d+),/', $html, $matches))
            $extra['ratingCount'] = (int) $matches[1];

        return $extra;
    }

    public function parseReviews()
    {
        $html = $this->dom->SaveHtml();
        $positive_texts = $negative_texts = array();

        if (preg_match_all('/"positiveText":"(.*?)"/', $html, $matches))
            $positive_texts = $matches[1];

        if (preg_match_all('/"negativeText":"(.*?)"/', $html, $matches))
            $negative_texts = $matches[1];

        if (!$positive_texts && !$negative_texts)
            return array();

        if (preg_match_all('/"averageScore":(\d+)/', $html, $matches))
            $ratings = $matches[1];

        if (preg_match_all('/"guestName":"(.+?)"/', $html, $matches))
            $authors = $matches[1];

        $results = array();

        for ($i = 0; $i < count($positive_texts); $i++)
        {
            $review = array();
            if (trim($positive_texts[$i]))
                $review['comment'] = "<p>[+] " . json_decode('"' . $positive_texts[$i] . '"') . "</p>";
            else
                $review['comment'] = '';

            if (isset($negative_texts[$i]) && $negative_texts[$i])
                $review['comment'] .= "\n<p>[-] " . json_decode('"' . $negative_texts[$i] . '"') . "</p>";

            if (isset($ratings[$i]))
            {
                $rating = round(TextHelper::convertRatingScale($ratings[$i], 1, 10, 1, 5), 2);
                $review['rating'] = TextHelper::ratingPrepare($rating);
            }

            if (isset($authors[$i]))
                $review['author'] = $authors[$i];

            $results[] = $review;
        }

        return $results;
    }

    public function getFeaturesXpath()
    {
        return array(
            array(
                'name' => ".//div[@id='hotelPoliciesInc']//div[not(contains(@class, 'children-policy'))]//p[@class='policy_name']",
                'value' => ".//div[@id='hotelPoliciesInc']/div[starts-with(@class, 'description')]/p[2]",
            ),
        );
    }

    public function parseFeatures()
    {
        $features = parent::parseFeatures();

        if ($location = $this->xpathScalar(".//meta[@name='twitter:title']/@content"))
        {
            $features[] = array(
                'name' => 'Location',
                'value' => $location,
            );
        }

        foreach ($features as $i => $f)
        {
            if (strlen($f['value']) > 90)
                unset($features[$i]);
            elseif ($f['name'] == 'Accepted payment methods')
                unset($features[$i]);
            elseif ($f['name'] == 'Cancellation prepayment')
                unset($features[$i]);
            elseif (strstr($f['name'], 'Cards accepted'))
                unset($features[$i]);
        }

        return $features;
    }

    public function getCurrency()
    {
        $currency = $this->xpathScalar(".//input[@name='selected_currency']/@value");

        if (!$currency)
        {
            if (preg_match("/b_selected_currency: '(\w+)'/ims", $this->dom->saveHtml(), $matches))
                $currency = $matches[1];
        }

        if (!$currency)
            $currency = 'USD';

        return $currency;
    }

    protected function _parseImages()
    {
        $images = array();

        $xpath = array(
            ".//div[@id='photo_wrapper']//img/@src",
            ".//div[contains(@class, 'bh-photo-grid')]//a/@data-thumb-url",
            ".//div[contains(@class, 'bh-photo-grid-thumbs-wrapper')]//a/img/@src",
        );

        $results = $this->xpathArray($xpath);
        foreach ($results as $img)
        {
            $img = preg_replace('~/max\d+/~', '/max1024x768/', $img);
            $images[] = $img;
        }

        return $images;
    }
}
