<?php

namespace Keywordrush\AffiliateEgg;

defined('\ABSPATH') || exit;

/**
 * EtsyParser class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */
class EtsyParser extends LdShopParser
{
    //protected $user_agent = array('Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:123.0) Gecko/20100101 Firefox/123.0');

    protected $charset = 'utf-8';
    protected $currency = 'USD';

    public function parseCatalog($max)
    {
        $locale = $this->extractLocale();
        $urls   = $this->getRawCatalogUrls();

        return array_slice(
            array_map(fn(string $url) => $this->applyLocale($url, $locale), $urls),
            0,
            $max
        );
    }

    private function getRawCatalogUrls(): array
    {
        // Try JSON source
        $jsonUrls = $this->parseCatalogJson();
        if (!empty($jsonUrls))
        {
            return $jsonUrls;
        }

        $paths = [
            "//a[contains(@class,'listing-link')]/@href",
        ];

        return $this->xpathArray($paths);
    }

    private function extractLocale(): ?string
    {
        $url    = $this->getUrl();
        $parsed = parse_url($url);

        // 1) Check 'locale' query parameter
        if (!empty($parsed['query']))
        {
            parse_str($parsed['query'], $params);
            if (!empty($params['locale']))
            {
                return $params['locale'];
            }
        }

        // 2) Fallback: detect two-letter locale in path, e.g. '/it/'
        if (!empty($parsed['path']) && preg_match('~^/([a-z]{2})(?=/|$)~i', $parsed['path'], $matches))
        {
            return strtolower($matches[1]);
        }

        return null;
    }

    private function applyLocale(string $url, ?string $locale): string
    {
        $base = strtok($url, '?');

        if ($locale === null)
        {
            return $base;
        }

        return $base . '?locale=' . urlencode($locale);
    }

    public function parseCatalogJson()
    {
        $json = $this->xpathScalar(".//script[@type='application/ld+json']");

        if (!$items = json_decode($json, true))
            return array();

        if (!isset($items['itemListElement']))
            return array();

        $urls = array();
        foreach ($items['itemListElement'] as $item)
        {
            if (isset($item['url']))
                $urls[] = $item['url'];

            //echo "\n\n" . $item['url'];
        }

        return $urls;
    }

    public function parseDescription()
    {
        $paths = array(
            ".//p[@class='wt-text-body-01 wt-break-word']",
        );

        return $this->xpathScalar($paths, true);
    }

    public function parsePrice()
    {
        $paths = array(
            ".//div[@data-buy-box-region='price']//p[@class='wt-text-title-03 wt-mr-xs-1']/span[2]",
            ".//div[@data-buy-box-region='price']//p[@class='wt-text-title-03 wt-mr-xs-1']",
            ".//div[@class='wt-mb-xs-3']//p[@class='wt-text-title-03 wt-mr-xs-2']",
            ".//*[@class='text-largest strong override-listing-price']",
            ".//p[@class='wt-text-title-03 wt-mr-xs-1']/span",
            ".//div[@data-selector='price-only']//p[@class='wt-text-title-larger wt-mr-xs-1 wt-text-slime ']",
        );

        if ($p = $this->xpathScalar($paths))
            return $p;

        if ($p = parent::parsePrice())
            return $p;
    }

    public function parseOldPrice()
    {
        $paths = array(
            ".//p[@class='wt-text-caption wt-text-gray']//span[@class='wt-text-strikethrough']",
            ".//p[contains(@class, 'wt-text-gray')]//span[@class='wt-text-strikethrough']",
            ".//div[@class='wt-mb-xs-3']//p[contains(@class, 'wt-text-strikethrough')]",
            ".//meta[@property='product:price:amount']/@content",

        );

        return $this->xpathScalar($paths);
    }

    public function parseImgLarge()
    {
        return str_replace('/il_570xN.', '/il_fullxfull.', $this->parseImg());
    }

    public function parseImg()
    {

        if (isset($this->ld_json['image'][0]['contentURL']))
        {
            $img = $this->ld_json['image'][0]['contentURL'];
            $img = str_replace('/il_fullxfull.', '/il_570xN.', $img);
            return $img;
        }

        return parent::parseImg();
    }

    public function getCurrency()
    {
        if ($this->parsePrice())
        {
            if (preg_match('/"locale_currency_code":"([A-Z]+?)"/', $this->dom->saveHtml(), $matches))
                return $matches[1];
        }
    }

    public function parseExtra()
    {
        $extra = parent::parseExtra();

        $extra['ratingDecimal'] = (float) str_replace(',', '.', $this->xpathScalar(".//span/input[@name='initial-rating']/@value"));
        $extra['images'] = $this->_parseImages();

        return $extra;
    }

    public function getFeaturesXpath()
    {
        return array(
            array(
                'name-value' => ".//p[@id='legacy-materials-product-details']",
                'separator' => ":",
            ),
        );
    }

    public function getReviewsXpath()
    {
        return array(
            array(
                'review' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//*[contains(@id, 'review-preview-toggle')]",
                'rating' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//span[@class='wt-display-inline-block wt-mr-xs-1']//span[last()]/@data-rating",
                'author' => ".//div[@id='same-listing-reviews-panel' or @id='reviews']//a[@class='wt-text-link wt-mr-xs-1']",
                'date' => ".//div[@id='same-listing-reviews-panel'or @id='reviews']//p[@class='wt-text-caption wt-text-gray']/text()",
            ),
        );
    }

    protected function _parseImages()
    {
        $paths = array(
            ".//div[contains(@class, 'image-carousel-container')]//*/@data-src",
            ".//div[contains(@class, 'image-carousel-container')]//*/@data-src-zoom-image",
            ".//div[contains(@data-selector, 'main-carousel')]//*/@data-zoom-src",
            ".//div[contains(@class, 'image-carousel-container')]//img/@src",
        );

        $images = (array) $this->xpathArray($paths);
        array_shift($images);

        foreach ($images as $i => $img)
        {
            if (strpos($img, 'il_fullxfull') !== false)
                $images[$i] = str_replace('/il_fullxfull.', '/il_570xN.', $img);
        }

        return $images;
    }
}
